package cn.virens.components.poi;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.virens.common.CloseableUtil;
import cn.virens.components.poi.listener.read.ReadCompleteListener;
import cn.virens.components.poi.listener.read.ReadLoadListener;
import cn.virens.components.poi.listener.read.ReadReadyListener;
import cn.virens.components.poi.read.CellReader;
import cn.virens.components.poi.read.RowValue;

/**
 * Excel 导入
 * 
 * @author : virens
 */
public class SimpleExcelImport {
	private static final Logger logger = LoggerFactory.getLogger(SimpleExcelImport.class);
	private static final DateFormat FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

	private int startRow = 0;
	private int endRow = Integer.MAX_VALUE;

	private List<CellReader> feilds = new ArrayList<>(); // 待读取的单元格（包含格式转换）

	private ReadCompleteListener completeListener;// 全都读取完成监听
	private ReadReadyListener readyListener;// 工作蒲读取监听
	private ReadLoadListener loadListener; // 行读取完成监听

	/**
	 * 开始导入
	 * 
	 * @param  filePath  表格文件路径
	 * @param  sheetAt   工作簿
	 * @throws Exception Exception
	 */
	public void imported(String filePath, int sheetAt) throws Exception {
		FileInputStream inputStream = new FileInputStream(filePath);
		XSSFWorkbook xssfWorkbook = new XSSFWorkbook(inputStream);// 输出表格
		try {
			XSSFSheet xssfSheet = xssfWorkbook.getSheetAt(sheetAt);

			if (xssfSheet != null) {
				this.imported0(xssfSheet);
			}
		} finally {
			CloseableUtil.close(xssfWorkbook);
			CloseableUtil.close(inputStream);
		}
	}

	/**
	 * 开始导入
	 * 
	 * @param  file      表格文件
	 * @param  sheetAt   工作簿
	 * @throws Exception Exception
	 */
	public void imported(File file, int sheetAt) throws Exception {
		FileInputStream inputStream = new FileInputStream(file);
		XSSFWorkbook xssfWorkbook = new XSSFWorkbook(inputStream);// 输出表格
		try {
			XSSFSheet xssfSheet = xssfWorkbook.getSheetAt(sheetAt);

			if (xssfSheet != null) {
				this.imported0(xssfSheet);
			}
		} finally {
			CloseableUtil.close(xssfWorkbook);
			CloseableUtil.close(inputStream);
		}
	}

	/**
	 * 开始导入
	 * 
	 * @param  inputStream 表格文件流
	 * @param  sheetAt     工作簿
	 * @throws Exception   Exception
	 */
	public void imported(InputStream inputStream, int sheetAt) throws Exception {
		XSSFWorkbook xssfWorkbook = new XSSFWorkbook(inputStream);// 输出表格
		try {
			XSSFSheet xssfSheet = xssfWorkbook.getSheetAt(sheetAt);

			if (xssfSheet != null) {
				this.imported0(xssfSheet);
			}
		} finally {
			CloseableUtil.close(xssfWorkbook);
			CloseableUtil.close(inputStream);
		}
	}

	/**
	 * 开始导入
	 * 
	 * @param  xssfSheet 工作簿
	 * @throws Exception Exception
	 */
	private void imported0(XSSFSheet xssfSheet) throws Exception {
		int _endRow = Math.min(endRow, xssfSheet.getLastRowNum());
		int _startRow = Math.max(startRow, xssfSheet.getFirstRowNum());

		// 如果有读取工作蒲的监听，则调用，并根据返回值判断是否继续读取
		if (readyListener == null || readyListener.apply(xssfSheet)) {
			final List<RowValue> rows = new ArrayList<>();

			logger.debug("开始导入数据...");
			/* 循环读取数据 */
			for (int index = _startRow; index <= _endRow; index++) {
				XSSFRow xssfRow = xssfSheet.getRow(index);// 获取行
				if (xssfRow == null) continue;

				RowValue rowObject = new RowValue(index);

				logger.debug("正在读取{}行...", index);

				/* 读取数据 */
				for (CellReader cellReader : feilds) {
					XSSFCell xssfCell = xssfRow.getCell(cellReader.getIndex());
					if (xssfCell == null) continue;

					try {
						CellType type = xssfCell.getCellType();
						if (type != null && type == CellType.ERROR) {
							rowObject.setCell(cellReader, null);
						} else if (type != null && type == CellType.BLANK) {
							rowObject.setCell(cellReader, "");
						} else if (type != null && type == CellType.FORMULA) {
							rowObject.setCell(cellReader, xssfCell.getCellFormula());
						} else if (type != null && type == CellType.STRING) {
							rowObject.setCell(cellReader, xssfCell.getStringCellValue());
						} else if (type != null && type == CellType.BOOLEAN) {
							rowObject.setCell(cellReader, xssfCell.getBooleanCellValue());
						} else if (type != null && type == CellType.NUMERIC && DateUtil.isCellDateFormatted(xssfCell)) {
							rowObject.setCell(cellReader, format(xssfCell.getDateCellValue()));
						} else if (type != null && type == CellType.NUMERIC) {
							rowObject.setCell(cellReader, String.valueOf(xssfCell.getNumericCellValue()));
						}
					} catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
				}

				// 回调读取成功处理
				if (loadListener == null || loadListener.apply(rowObject)) {
					rows.add(rowObject);
				}
			}

			logger.debug("一共导入了{}条数据!", rows.size());
			logger.debug("导入完成!");

			// 如果有读取完成监听，就将全部行一起发给该监听
			if (completeListener != null) {
				completeListener.accept(rows);
			}
		} else {
			throw new Exception("Excel未准备");
		}
	}

	/**
	 * 获取 数据起始行
	 * 
	 * @return startRow
	 */
	public int getStartRow() {
		return startRow;
	}

	/**
	 * 设置 数据起始行
	 * 
	 * @param startRow 数据起始行
	 */
	public void setStartRow(int startRow) {
		this.startRow = startRow;
	}

	/**
	 * 获取 数据截止行
	 * 
	 * @return endRow
	 */
	public int getEndRow() {
		return endRow;
	}

	/**
	 * 设置 数据截止行
	 * 
	 * @param endRow 数据截止行
	 */
	public void setEndRow(int endRow) {
		this.endRow = endRow;
	}

	/**
	 * 获取 准备监听
	 * 
	 * @return readListener
	 */
	public ReadLoadListener getLoadListener() {
		return loadListener;
	}

	/**
	 * 设置 准备监听
	 * 
	 * @param readListener 准备监听
	 */
	public void setLoadListener(ReadLoadListener readListener) {
		this.loadListener = readListener;
	}

	/**
	 * 获取 导入完成监听
	 * 
	 * @return readCompleteListener
	 */
	public ReadCompleteListener getCompleteListener() {
		return completeListener;
	}

	/**
	 * 设置 导入完成监听
	 * 
	 * @param readCompleteListener 导入完成监听
	 */
	public void setCompleteListener(ReadCompleteListener readCompleteListener) {
		this.completeListener = readCompleteListener;
	}

	/**
	 * 获取 导入监听
	 * 
	 * @return readyListener
	 */
	public ReadReadyListener getReadyListener() {
		return readyListener;
	}

	/**
	 * 设置 导入监听
	 * 
	 * @param readyListener 导入监听
	 */
	public void setReadyListener(ReadReadyListener readyListener) {
		this.readyListener = readyListener;
	}

	/**
	 * 添加 读取字段
	 * 
	 * @param  readCell 读取字段
	 * @return          boolean
	 */
	public boolean addFeild(CellReader readCell) {
		return this.feilds.add(readCell);
	}

	private static String format(Date date) {
		return date == null ? null : FORMAT.format(date);
	}
}
