package com.amity.socialcloud.sdk

import com.amity.socialcloud.sdk.core.AmityConnectionState
import com.amity.socialcloud.sdk.core.AmityGlobalBanEvent
import com.amity.socialcloud.sdk.core.authen.UserRegistration
import com.amity.socialcloud.sdk.core.domain.notification.RegisterDeviceForNotificationUseCase
import com.amity.socialcloud.sdk.core.domain.notification.UnregisterAllUsersForNotificationUseCase
import com.amity.socialcloud.sdk.core.domain.notification.UnregisterDeviceForNotificationUseCase
import com.amity.socialcloud.sdk.core.domain.session.ActiveUserIdGetUseCase
import com.amity.socialcloud.sdk.core.domain.session.LogoutUseCase
import com.amity.socialcloud.sdk.core.encryption.AmityDBEncryption
import com.amity.socialcloud.sdk.core.events.AmityTopic
import com.amity.socialcloud.sdk.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.core.file.AmityFileRepository
import com.amity.socialcloud.sdk.core.permission.AmityPermission
import com.amity.socialcloud.sdk.core.permission.AmityPermissionValidator
import com.amity.socialcloud.sdk.core.session.SessionError
import com.amity.socialcloud.sdk.core.session.SessionHandler
import com.amity.socialcloud.sdk.core.session.SessionStateManager
import com.amity.socialcloud.sdk.core.session.component.DatabaseSessionComponent
import com.amity.socialcloud.sdk.core.session.component.TokenRenewalSessionComponent
import com.amity.socialcloud.sdk.core.session.component.TokenWatcherSessionComponent
import com.amity.socialcloud.sdk.core.session.component.UserSettingSessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.amity.socialcloud.sdk.core.user.AmityUserNotification
import com.amity.socialcloud.sdk.core.user.AmityUserRepository
import com.amity.socialcloud.sdk.core.user.AmityUserUpdate
import com.amity.socialcloud.sdk.infra.CoreClient
import com.amity.socialcloud.sdk.infra.mqtt.AmityMqttClient
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.streamapi.AmityClientConfiguration
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.AmityContentCheck
import com.ekoapp.ekosdk.StreamFunction
import com.ekoapp.ekosdk.internal.api.EkoApi
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoApiKey
import com.ekoapp.ekosdk.internal.data.model.EkoHttpUrl
import com.ekoapp.ekosdk.internal.data.model.EkoMqttUrl
import com.ekoapp.ekosdk.internal.data.model.EkoSocketUrl
import com.ekoapp.ekosdk.internal.init.AmityCoreSDKInitializer
import com.ekoapp.ekosdk.internal.init.EkoSdkInitProvider
import com.ekoapp.ekosdk.internal.usecase.content.GetContentCheckUseCase
import com.ekoapp.ekosdk.internal.util.AppContext
import com.ekoapp.ekosdk.internal.util.EkoPreconditions
import com.ekoapp.ekosdk.sdk.BuildConfig
import com.google.common.base.Objects
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Maybe
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.CompletableSubject
import org.joda.time.DateTime

object AmityCoreClient {

    internal var millisTimeDiff = 0
    internal fun getServerTime(): DateTime {
        return DateTime.now().plusMillis(millisTimeDiff)
    }

    fun getCurrentSessionState(): SessionState {
        return CoreClient.sessionStateManager!!.sessionState
    }
    /**
     * This method setup the SDK by overriding default endpoints
     *
     * @param apiKey an api key to be used with the SDK
     * @param endpoint an endpoint model to override default endpoints
     */
    fun setup(
        apiKey: String,
        endpoint: AmityEndpoint = AmityEndpoint.SG,
        dbEncryption: AmityDBEncryption = AmityDBEncryption.NONE
    ): Completable {
        return CoreClient.setup(apiKey, endpoint, dbEncryption)
    }

    /**
     * This method setup the SDK by overriding the respective default endpoints with the provided http and socket endpoints
     *
     * @param apiKey an api key to be used with the SDK
     * @param httpEndpoint a custom http endpoint to override default http endpoint
     * @param socketEndpoint a custom socket endpoint to override default socket endpoint
     */
    @Deprecated("please use setup(apiKey: String, endpoint: AmityEndpoint) instead")
    fun setup(apiKey: String, httpEndpoint: String, socketEndpoint: String): Completable {
        val mqttEndpoint = when (httpEndpoint) {
            EU_HTTP_ENDPOINT -> EU_MQTT_ENDPOINT
            US_HTTP_ENDPOINT -> US_MQTT_ENDPOINT
            else -> SG_MQTT_ENDPOINT
        }
        return setup(apiKey, AmityEndpoint.CUSTOM(httpEndpoint, socketEndpoint, mqttEndpoint))
    }

    /**
     * This method setup the SDK with default endpoints
     *
     * @param apiKey an api key to be used with the SDK
     */
    fun setup(apiKey: String): Completable {
        return setup(apiKey, AmityEndpoint.SG)
    }

    fun registerDeviceForPushNotification(): Completable {
        return RegisterDeviceForNotificationUseCase().execute().toV2()
    }

    fun unregisterDeviceForPushNotification(): Completable {
        return UnregisterAllUsersForNotificationUseCase().execute().toV2()
    }

    fun unregisterDeviceForPushNotification(userId: String): Completable {
        return UnregisterDeviceForNotificationUseCase().execute(userId).toV2()
    }

    fun disconnect(): Completable {
        return if (CoreClient.mqttClient == null || CoreClient.socketClient == null) {
            Completable.complete()
        } else {
            CoreClient.socketClient?.disconnect()
            CoreClient.mqttClient!!.disconnect()
        }
    }

    @Deprecated(
        "Please use login function with sessionHandler instead", ReplaceWith(
            "login(userId = userId, sessionHandler = sessionHandler)",
            "com.amity.socialcloud.sdk.AmityCoreClient.login"
        )
    )
    fun login(userId: String): UserRegistration.Builder {
        return login(userId = userId, sessionHandler = null)
    }

    fun login(userId: String, sessionHandler: SessionHandler?): UserRegistration.Builder {
        val isLegacyVersion = sessionHandler == null
        val currentSessionState = getCurrentSessionState()
        if (currentSessionState is SessionState.Establishing) {
            throw SessionError.fromState(currentSessionState)
        }

        sessionHandler?.let {
            CoreClient.renewalManager?.sessionHandler = it
        }

        return UserRegistration.Builder(
            appEventBus = CoreClient.appEventBus!!,
            sessionLifeCycleEventBus = CoreClient.sessionLifeCycleEventBus!!,
            userId = userId,
            isLegacyVersion = isLegacyVersion
        )
    }

    fun logout(): Completable {
        // FIXME: check all of the prerequisites
        CoreClient.appEventBus!!.publish(AppEvent.ManualLogout)
        return LogoutUseCase().execute().toV2()
    }

    fun getUserId(): String {
        return ActiveUserIdGetUseCase().execute()
    }

    fun updateUser(): AmityUserUpdate.Builder {
        return AmityUserUpdate.Builder(getUserId())
    }

    fun hasPermission(permission: AmityPermission): AmityPermissionValidator {
        return AmityPermissionValidator(permission)
    }

    fun getCurrentUser(): Flowable<AmityUser> {
        return newUserRepository().getCurrentUser()
    }

    fun newUserRepository(): AmityUserRepository {
        return AmityUserRepository()
    }

    fun newFileRepository(): AmityFileRepository {
        return AmityFileRepository()
    }

    fun getAmityCoreSdkVersion(): String {
        return BuildConfig.VERSION_NAME
    }

    fun getGlobalBanEvents(): Flowable<AmityGlobalBanEvent> {
        return EkoSocket.globalBanEvents
    }

    fun notification(): AmityUserNotification {
        return AmityUserNotification()
    }

    fun getConfiguration(): AmityClientConfiguration {
        return AmityClientConfiguration(StreamFunction())
    }

    @Deprecated("Please use  observeSessionState instead")
    fun getConnectionState(): Flowable<AmityConnectionState> {
        return EkoSocket.connectionState
    }

    fun subscription(topic: AmityTopic): AmityTopicSubscription {
        return AmityTopicSubscription(topic)
    }

    fun getContentCheck(): Single<AmityContentCheck> {
        return GetContentCheckUseCase()
            .execute()
    }

    fun observeSessionState(): Flowable<SessionState> {
        return CoreClient.sessionStateEventBus!!.observe().toV2()
    }


}