package com.amity.socialcloud.sdk.chat.data.channel

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.chat.channel.*
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRepository
import com.amity.socialcloud.sdk.chat.data.channel.paging.ChannelMediator
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.core.error.AmityError
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.AmityObjectRepository
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.api.dto.ChannelQueryDto
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoChannelBoundaryCallback
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.channel.ChannelUpdateOption
import com.ekoapp.ekosdk.internal.repository.channel.EkoChannelFactory
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.subjects.PublishSubject

@OptIn(ExperimentalPagingApi::class)
internal class ChannelRepository : AmityObjectRepository<EkoChannelEntity, AmityChannel>() {

    override fun fetchAndSave(objectId: String): Completable {
        return ChannelRemoteDataStore().getChannel(objectId)
            .flatMapCompletable {
                persistChannels(it)
            }
    }

    override fun queryFromCache(objectId: String): EkoChannelEntity? {
        return ChannelLocalDataStore().getChannel(objectId)
    }

    override fun mapper(): ModelMapper<EkoChannelEntity, AmityChannel> {
        return ChannelModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EkoChannelEntity> {
        return ChannelLocalDataStore().observeChannel(objectId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun getChannelPagedList(
        isDeleted: Boolean?,
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        sortBy: AmityChannelSortOption
    ): Flowable<PagedList<AmityChannel>> {

        val factory =
            EkoChannelFactory(isDeleted, types, filter, includingTags, excludingTags, sortBy)

        val delaySubject = PublishSubject.create<Boolean>()
        val boundaryCallback = EkoChannelBoundaryCallback(
            factory.getChannelType(),
            filter,
            isDeleted,
            includingTags,
            excludingTags,
            getDefaultPageSize(),
            delaySubject
        )

        return createRxCollectionWithBoundaryCallback(
            factory.getDataSource().map(boundaryCallback), boundaryCallback
        ).toV3()
    }

    fun getChannelPagingData(
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
    ): Flowable<PagingData<AmityChannel>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                prefetchDistance = 0,
                initialLoadSize = (getDefaultPageSize()/2),
            ),
            dynamicQueryStreamMediator = ChannelMediator(
                types = types,
                filter = filter,
                includingTags = includingTags,
                excludingTags = excludingTags,
                isDeleted = isDeleted,
            ),
            pagingSourceFactory = {
                ChannelLocalDataStore().getChannelPagingSource(
                    types = types,
                    filter = filter,
                    includingTags = includingTags,
                    excludingTags = excludingTags,
                    isDeleted = isDeleted,
                )
            },
            modelMapper = ChannelModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun createChannel(
        channelId: String? = null,
        creationType: AmityChannelCreator.CreationType,
        option: ChannelCreateOption
    ): Single<AmityChannel> {
        return ChannelRemoteDataStore().createChannel(
            channelId = channelId,
            creationType = creationType,
            displayName = option.displayName,
            avatarFileId = option.avatarFileId,
            metadata = option.metadata,
            userIds = option.userIds,
            tags = option.tags
        ).flatMap {
                persistAndReturnChannel(it)
            }
    }

    fun createConversationChannel(
        option: ChannelCreateOption
    ): Single<AmityChannel> {
        return ChannelRemoteDataStore().createConversationChannel(
            displayName = option.displayName,
            avatarFileId = option.avatarFileId,
            metadata = option.metadata,
            userIds = option.userIds,
            tags = option.tags
        ).flatMap {
                persistAndReturnChannel(it)
            }
    }

    fun joinChannel(channelId: String): Single<AmityChannel> {
        return ChannelRemoteDataStore().joinChannel(channelId)
            .flatMap {
                persistAndReturnChannel(it)
            }
            .onErrorResumeNext {
                if (AmityError.USER_IS_BANNED == AmityError.from(it)) {
                    ChannelMembershipRepository().handleMembershipBanned(
                        channelId,
                        AmityCoreClient.getUserId()
                    )
                        .andThen(Single.error(it))
                } else {
                    Single.error(it)
                }
            }
    }

    fun leaveChannel(channelId: String): Completable {
        return ChannelRemoteDataStore().leaveChannel(channelId)
            .flatMapCompletable {
                persistChannels(it)
            }
    }

    fun fetchChannel(channelId: String): Completable {
        return ChannelRemoteDataStore().getChannel(channelId)
            .flatMapCompletable {
                persistChannels(it)
            }
    }

    fun observeChannel(channelId: String): Flowable<AmityChannel> {
        return UserDatabase.get().channelDao().getChannel(channelId)
            .map {
                ChannelModelMapper().map(it)
            }
    }

    fun updateChannel(
        channelId: String,
        option: ChannelUpdateOption
    ): Single<AmityChannel> {
        return ChannelRemoteDataStore().updateChannel(
            channelId = channelId,
            displayName = option.displayName,
            avatarFileId = option.avatarFileId,
            metadata = option.metadata,
            tags = option.tags
        ).flatMap {
                persistAndReturnChannel(it)
            }
    }

    fun getLatestChannel(
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<AmityChannel> {
        return ChannelLocalDataStore().getLatestChannel(
            types = types,
            filter = filter,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                ChannelModelMapper().map(it)
            }
    }

    fun getAllJoinedChannels() : Flowable<List<AmityChannel>> {
        return ChannelLocalDataStore().getAllJoinedChannels()
            .map {
                it.map {
                    ChannelModelMapper().map(it)
                }
            }
    }

    internal fun persistChannels(dto: ChannelQueryDto): Completable {
        return ChannelQueryPersister().persist(dto)
    }

    internal fun getChannel(channelId: String): Single<AmityChannel> {
        return ChannelLocalDataStore().observeChannel(channelId)
            .firstOrError()
            .map {
                ChannelModelMapper().map(it)
            }
    }

    internal fun persistAndReturnChannel(dto: ChannelQueryDto): Single<AmityChannel> {
        return persistChannels(dto).andThen(getChannel(dto.channelDtoList?.first()?.channelId!!))
    }
}