package com.amity.socialcloud.sdk.chat.data.channel.membership

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembershipSortOption
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.UserLocalDataStore
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.core.user.AmityMembershipType
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.entity.ChannelMembershipEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime

class ChannelMembershipLocalDataStore {

    fun saveChannelMembers(
        members: List<ChannelMembershipEntity>,
        channelIds: List<String>
    ): Completable {
        return Completable.fromAction {
            UserDatabase.get().channelMembershipDao().save(members)
            createMissingMemberships(members, channelIds).takeIf { it.isNotEmpty() }?.let {
                UserDatabase.get().channelMembershipDao().save(it)
            }
        }
    }

    private fun createMissingMemberships(
        members: List<ChannelMembershipEntity>,
        channelIds: List<String>
    ): List<ChannelMembershipEntity> {
        val missingMembers = mutableListOf<ChannelMembershipEntity>()
        channelIds.forEach { channelId ->
            val myUserId = SessionLocalDataStore().getActiveUserId()
            var myMember = members.find { member ->
                member.channelId == channelId && member.userId == myUserId
            }
            val hasCachedMembership = getChannelMember(channelId, myUserId) != null
            if (myMember == null && !hasCachedMembership) {
                myMember = ChannelMembershipEntity.create(channelId, myUserId)
                missingMembers.add(myMember)
            }
        }
        return missingMembers
    }

    fun getChannelMembershipPagingSource(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        sortOption: AmityChannelMembershipSortOption?
    ): PagingSource<Int, ChannelMembershipEntity> {
        return UserDatabase.get().channelMembershipPagingDao().getChannelMembershipPagingSource(
            channelId = channelId,
            memberships = memberships,
            isMuted = isMuted,
            roles = roles,
            sortOption = sortOption
        )
    }

    fun getChannelMemberSearchPagingSource(
        channelId: String,
        keyword: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
    ): PagingSource<Int, ChannelMembershipEntity> {
        return UserDatabase.get().channelMemberSearchPagingDao().getChannelMembershipPagingSource(
            channelId = channelId,
            keyword = keyword,
            memberships = memberships,
            isMuted = isMuted,
            roles = roles,
        )
    }

    fun getLatestChannel(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<ChannelMembershipEntity> {
        return UserDatabase.get().channelMembershipDao().getLatestChannelMembership(
                channelId,
                memberships.map { it.apiKey },
                isMuted,
                roles,
                dynamicQueryStreamKeyCreator.toMap().hashCode(),
                nonce,
                DateTime.now()
            )
    }

    fun getChannelMember(channelId: String, userId: String): ChannelMembershipEntity? {
        return UserDatabase.get().channelMembershipDao().getByChannelIdAndUserIdNow(channelId, userId)
    }

    fun observeChannelMember(channelId: String, userId: String): Flowable<ChannelMembershipEntity> {
        return UserDatabase.get().channelMembershipDao().getByChannelIdAndUserId(channelId, userId)
    }

    fun updateMembership(channelId: String, userId: String, membership: AmityMembershipType) {
        return  UserDatabase.get().channelMembershipDao().updateMembership(channelId, userId, membership.apiKey)
    }

    fun notifyUserUpdate(userId: String) {
        UserDatabase.get().channelMembershipDao().updateUser(userId)
    }

}