package com.amity.socialcloud.sdk.chat.data.channel.membership

import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.internal.api.EkoApi
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.dto.ChannelMembershipQueryDto
import com.ekoapp.ekosdk.internal.api.dto.EkoChannelMembershipPrivilegesListDto
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.ChannelMemberPermissionsConverter
import com.ekoapp.ekosdk.internal.api.socket.request.*
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import org.joda.time.Duration

class ChannelMembershipRemoteDataStore {

    fun queryChannelMembers(
        channelId: String,
        keyword: String? = null,
        roles: List<String>? = null,
        memberships: List<String>? = null,
        sortBy: String? = null,
        options: QueryOptionsRequestParams
    ): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.queryChannelUsers(
                    channelId = channelId,
                    keyword = keyword,
                    roles = roles?.takeIf { it.isNotEmpty() },
                    memberships = memberships?.takeIf { it.isNotEmpty() },
                    sortBy = sortBy,
                    limit = options.limit,
                    token = options.token
                )
            }
    }

    fun addChannelMembers(
        channelId: String,
        userIds: List<String>
    ): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.addChannelUsers(channelId, ChannelAddUsersRequest(channelId, userIds))
            }
    }

    fun removeChannelMembers(
        channelId: String,
        userIds: List<String>
    ): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.deleteChannelUsers(channelId, ChannelRemoveUsersRequest(channelId, userIds))
            }
    }

    fun assignChannelRole(channelId: String, role: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.assignChannelRole(channelId, ChannelAddRoleRequest(channelId, role, userIds))
            }
    }

    fun unassignChannelRole(channelId: String, role: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.unassignChannelRole(channelId, ChannelRemoveRoleRequest(channelId, role, userIds))
            }
    }

    fun banChannelMembers(channelId: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.banChannelUsers(channelId, ChannelBanUserRequest(userIds))
            }
    }

    fun unbanChannelMembers(channelId: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMap {
                it.unbanChannelUsers(channelId, ChannelUnbanUserRequest(channelId, userIds))
            }
    }

    fun muteChannel(channelId: String, timeout: Duration): Completable {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMapCompletable {
                it.muteChannel(channelId, ChannelSetMutedRequest(channelId, timeout.millis))
            }
    }

    fun muteChannelMembers(channelId: String, timeout: Duration, userIds: List<String>): Completable {
        return EkoApi.get(ChannelMembershipApi::class)
            .flatMapCompletable {
                it.muteChannelUser(channelId, ChannelSetMutedUsersRequest(channelId, timeout.millis, userIds))
            }
    }

    fun getPrivileges(channelId: String): Single<EkoChannelMembershipPrivilegesListDto> {
        val request = ChannelMembershipGetPermissionsRequest(channelId)
        val converter = ChannelMemberPermissionsConverter()
        return EkoSocket.call(Call.create(request, converter)).toV3()
    }

}