package com.amity.socialcloud.sdk.chat.data.channel.membership

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMember
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembershipPrivileges
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembershipSortOption
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.channel.membership.paging.ChannelMemberSearchMediator
import com.amity.socialcloud.sdk.chat.data.channel.membership.paging.ChannelMembershipMediator
import com.amity.socialcloud.sdk.chat.data.channel.reader.ChannelReaderLocalDataStore
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.UserLocalDataStore
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.core.user.AmityMembershipType
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.dto.ChannelMembershipQueryDto
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoChannelMembershipBoundaryCallback
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import org.joda.time.Duration

@OptIn(ExperimentalPagingApi::class)
internal class ChannelMembershipRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun observeChannelMember(channelId: String, userId: String): Flowable<AmityChannelMember> {
        return ChannelMembershipLocalDataStore().observeChannelMember(channelId, userId)
            .map {
                ChannelMembershipModelMapper().map(it)
            }
    }

    fun addChannelMembers(channelId: String, userIds: List<String>): Single<AmityChannel> {
        return ChannelMembershipRemoteDataStore().addChannelMembers(channelId, userIds)
            .flatMap {
                persistAndReturnChannel(it)
            }
    }

    fun removeChannelMembers(channelId: String, userIds: List<String>): Single<AmityChannel> {
        return ChannelMembershipRemoteDataStore().removeChannelMembers(channelId, userIds)
            .flatMap {
                persistAndReturnChannel(it)
            }
    }

    fun getChannelMembershipPagedList(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        sortOption: AmityChannelMembershipSortOption
    ): Flowable<PagedList<AmityChannelMember>> {

        val channelMembershipDao = UserDatabase.get().channelMembershipDao()
        val factory = channelMembershipDao.getDataSource(
            channelId,
            roles,
            memberships.map { it.apiKey },
            isMuted,
            sortOption
        )

        return createRxCollectionWithBoundaryCallback(
            factory.map(ChannelMembershipComposerFunction()),
            EkoChannelMembershipBoundaryCallback(
                channelId = channelId,
                memberships = memberships,
                isMuted = isMuted,
                roles = roles,
                sortOption = sortOption,
                pageSize = DEFAULT_PAGE_SIZE
            )
        ).toV3()
    }

    fun getChannelMembershipPagingData(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        sortOption: AmityChannelMembershipSortOption
    ): Flowable<PagingData<AmityChannelMember>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = ChannelMembershipMediator(
                channelId = channelId,
                memberships = memberships,
                isMuted = isMuted,
                roles = roles,
                sortOption = sortOption
            ),
            pagingSourceFactory = {
                ChannelMembershipLocalDataStore().getChannelMembershipPagingSource(
                    channelId = channelId,
                    memberships = memberships,
                    isMuted = isMuted,
                    roles = roles,
                    sortOption = sortOption
                )
            },
            modelMapper = ChannelMembershipModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun searchChannelMembershipPagingData(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        keyword: String
    ): Flowable<PagingData<AmityChannelMember>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = ChannelMemberSearchMediator(
                channelId = channelId,
                keyword = keyword,
                memberships = memberships,
                isMuted = isMuted,
                roles = roles,
            ),
            pagingSourceFactory = {
                ChannelMembershipLocalDataStore().getChannelMemberSearchPagingSource(
                    channelId = channelId,
                    keyword = keyword,
                    memberships = memberships,
                    isMuted = isMuted,
                    roles = roles
                )
            },
            modelMapper = ChannelMembershipModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun assignChannelRole(channelId: String, role: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().assignChannelRole(channelId, role, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun unassignChannelRole(channelId: String, role: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().unassignChannelRole(channelId, role, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun banChannelUsers(channelId: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().banChannelMembers(channelId, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun unbanChannelUsers(channelId: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().unbanChannelMembers(channelId, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun muteChannelUsers(channelId: String, timeout: Duration, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().muteChannelMembers(channelId, timeout, userIds)
    }

    fun unmuteChannelUsers(channelId: String, userIds: List<String>): Completable {
        return muteChannelUsers(channelId, Duration.ZERO, userIds)
    }

    fun muteChannel(channelId: String, timeout: Duration): Completable {
        return ChannelMembershipRemoteDataStore().muteChannel(channelId, timeout)
    }

    fun unmuteChannel(channelId: String): Completable {
        return muteChannel(channelId, Duration.ZERO)
    }

    fun handleMembershipBanned(channelId: String, userId: String) : Completable {
        return Completable.fromAction {
            if (SessionLocalDataStore().getActiveUserId() == userId) {
                ChannelReaderLocalDataStore().stopReading(channelId)
            }
            ChannelMembershipLocalDataStore().updateMembership(channelId, userId, AmityMembershipType.BANNED)
            UserDatabase.get().messageDao().softDeleteFromChannelByUserId(channelId, userId)
        }
    }

    fun getPrivileges(channelId: String): Single<AmityChannelMembershipPrivileges> {
        return ChannelMembershipRemoteDataStore().getPrivileges(channelId)
            .map {
                it.results
            }
    }

    fun getLatestChannel(
        channelId: String,
        roles: AmityRoles,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<AmityChannelMember> {
        return ChannelMembershipLocalDataStore().getLatestChannel(
            channelId = channelId,
            memberships = memberships,
            isMuted = isMuted,
            roles = roles,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                ChannelMembershipModelMapper().map(it)
            }
    }

    internal fun persistChannelMemberships(dto: ChannelMembershipQueryDto): Completable {
        return ChannelMembershipQueryPersister().persist(dto)
    }

    // FIXME: Doesn't make sense to return channel from member API. Review this for the next major version
    internal fun persistAndReturnChannel(dto: ChannelMembershipQueryDto): Single<AmityChannel> {
        return persistChannelMemberships(dto).andThen(ChannelRepository().getChannel(dto.channelDtoList.first().channelId))
    }

}