package com.amity.socialcloud.sdk.chat.data.channel.membership.paging

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMember
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembership
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembershipSortOption
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipQueryPersister
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRepository
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.core.user.AmityMembershipType
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.ChannelMembershipQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.ChannelMembershipEntity
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single


@OptIn(ExperimentalPagingApi::class)
internal class ChannelMembershipMediator(
    private val channelId: String,
    private val memberships: List<AmityMembershipType>,
    private val isMuted: Boolean?,
    private val roles: AmityRoles,
    private val sortOption: AmityChannelMembershipSortOption?
) : DynamicQueryStreamMediator<ChannelMembershipEntity, ChannelMembershipQueryDto, AmityChannelMember>(

    nonce = AmityNonce.CHANNEL_MEMBERSHIP,
    dynamicQueryStreamKeyCreator = ChannelMembershipKeyCreator(
        channelId = channelId,
        memberships = memberships,
        isMuted = isMuted,
        roles = roles,
        sortOption = sortOption
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityChannelMember> {
        return ChannelMembershipRepository().getLatestChannel(
            channelId = channelId,
            memberships = memberships,
            isMuted = isMuted,
            roles = roles,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toV2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<ChannelMembershipQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<ChannelMembershipQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: ChannelMembershipQueryDto): Completable {
        return ChannelMembershipQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: ChannelMembershipQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = dto.channelMembershipDtoList?.map { it.userId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<ChannelMembershipQueryDto> {
        return ChannelMembershipRemoteDataStore().queryChannelMembers(
            channelId = channelId,
            roles = roles.get(),
            memberships = generateMembershipParams(memberships, isMuted),
            sortBy  = sortOption?.apiKey,
            options = QueryOptionsRequestParams(limit, token)
        ).toV2()
    }

    private fun generateMembershipParams(memberships: List<AmityMembershipType>, isMuted: Boolean?) : List<String> {
        return memberships.map { it.apiKey }.plus(if(isMuted != null) listOf(AmityChannelMembership.MUTED.apiKey) else emptyList())
    }

}