package com.amity.socialcloud.sdk.chat.data.channel.paging

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.chat.channel.AmityChannelFilter
import com.amity.socialcloud.sdk.chat.data.channel.ChannelHelper
import com.amity.socialcloud.sdk.chat.data.channel.ChannelQueryPersister
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.api.dto.ChannelQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single


@OptIn(ExperimentalPagingApi::class)
internal class ChannelMediator(
    private val types: Set<AmityChannel.Type>,
    private val filter: AmityChannelFilter,
    private val includingTags: AmityTags,
    private val excludingTags: AmityTags,
    private val isDeleted: Boolean?,
) : DynamicQueryStreamMediator<EkoChannelEntity, ChannelQueryDto, AmityChannel>(

    nonce = AmityNonce.CHANNEL_LIST,
    dynamicQueryStreamKeyCreator = ChannelKeyCreator(
        types = types,
        filter = filter,
        includingTags = includingTags,
        excludingTags = excludingTags,
        isDeleted = isDeleted
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityChannel> {
        return ChannelRepository().getLatestChannel(
            types = types,
            filter = filter,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator =  dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toV2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<ChannelQueryDto> {
        return getRequest(limit = pageSize)

    }

    override fun getFetchByTokenRequest(token: String): Single<ChannelQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: ChannelQueryDto): Completable {
        return ChannelQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: ChannelQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = dto.channelDtoList?.map { it.channelId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<ChannelQueryDto> {
        return ChannelRemoteDataStore().queryChannels(
            types = ChannelHelper.getChannelTypes(types),
            filter = filter.apiKey,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(limit, token)
        ).toV2()
    }

}