package com.amity.socialcloud.sdk.chat.data.channel.reader

import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.chat.data.channel.ChannelLocalDataStore
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipLocalDataStore
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRepository
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.UserLocalDataStore
import com.amity.socialcloud.sdk.core.error.AmityError
import com.ekoapp.ekosdk.EkoChannelReadStatus
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.google.common.base.Objects
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class ChannelReaderRepository {

    fun startReading(channelId: String) {
        val localDataStore = ChannelReaderLocalDataStore()
        val startReading = localDataStore.startReading(channelId)
            .concatWith(ChannelReaderRemoteDataStore().startReading(channelId))

        localDataStore.observeChannelReader(channelId)
            .firstElement()
            .filter { !Objects.equal(EkoChannelReadStatus.READING, it.readStatus) }
            .flatMapCompletable { startReading }
            .onErrorResumeNext {
                if (AmityError.USER_IS_BANNED == AmityError.from(it)) {
                    ChannelMembershipRepository().handleMembershipBanned(
                        channelId,
                        AmityCoreClient.getUserId()
                    ).andThen(Completable.error(it))
                } else {
                    Completable.error(it)
                }
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    fun stopReading(channelId: String) {
        val localDataStore = ChannelReaderLocalDataStore()
        val stopReading = localDataStore.stopReading(channelId)
            .concatWith(ChannelReaderRemoteDataStore().stopReading(channelId))

        localDataStore.observeChannelReader(channelId)
            .firstElement()
            .filter { !Objects.equal(EkoChannelReadStatus.NOT_READING, it.readStatus) }
            .flatMapCompletable { stopReading }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    fun getUnreadCount(channelId: String): Int {
        val messageCount = ChannelLocalDataStore().getChannel(channelId)?.messageCount ?: 0
        val myUserId = SessionLocalDataStore().getActiveUserId()
        val channelReadToSegment = ChannelMembershipLocalDataStore().getChannelMember(channelId, myUserId)?.readToSegment ?: 0
        val localReadToSegment = ChannelReaderLocalDataStore().getChannelReader(channelId)?.getLocalReadToSegment() ?: 0
        return Math.max(0, messageCount - Math.max(channelReadToSegment, localReadToSegment))
    }

    fun hasUnreadMention(channelId: String): Boolean {
        val myUserId = SessionLocalDataStore().getActiveUserId()

        val channelMember = ChannelMembershipLocalDataStore().getChannelMember(channelId, myUserId)
        val channelReadToSegment = channelMember?.readToSegment ?: 0
        val channelLastMentionedSegment = channelMember?.lastMentionedSegment ?: 0

        val channelReader = ChannelReaderLocalDataStore().getChannelReader(channelId)
        val localReadToSegment = channelReader?.getLocalReadToSegment() ?: 0
        val localLastMentionedSegment = channelReader?.localLastMentionedSegment ?: 0

        val readToSegment = Math.max(channelReadToSegment, localReadToSegment)
        val lastMentionedSegment: Int = Math.max(channelLastMentionedSegment, localLastMentionedSegment)

        return readToSegment < lastMentionedSegment
    }

}