package com.amity.socialcloud.sdk.chat.data.message

import android.net.Uri
import androidx.annotation.WorkerThread
import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.chat.channel.AmityChannelFilter
import com.amity.socialcloud.sdk.chat.message.AmityMessage
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.core.mention.AmityMentioneeTarget
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoMessageFlag
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class MessageLocalDataStore {

    fun saveMessages(
        messages: List<EkoMessageEntity>,
    ): Completable {
        return Completable.fromAction {
            UserDatabase.get().messageDao().save(messages)
        }.subscribeOn(Schedulers.io())
    }

    fun createMessage(
        channelId: String,
        parentId: String?,
        type: String,
        data: JsonObject?,
        metadata: JsonObject?,
        tags: AmityTags,
        fileUri: Uri?,
        mentionees: List<AmityMentioneeTarget>
    ): Single<EkoMessageEntity> {

        return Single.fromCallable {
            val message = EkoMessageEntity()
            message.channelId = channelId
            message.userId = AmityCoreClient.getUserId()
            message.parentId = parentId
            message.type = type
            message.data = data
            message.metadata = metadata
            message.tags = tags
            message.mentionees = MessageMentionMapper().map(mentionees)
            message.setState(AmityMessage.State.CREATED)
            val now = DateTime()
            message.createdAt = now
            message.editedAt = now
            message.updatedAt = now

            fileUri?.let {
                message.fileId = message.messageId
            }

            val messageDao = UserDatabase.get().messageDao()
            val highestChannelSegment = messageDao.getHighestChannelSegment(message.channelId) + 1
            message.setChannelSegment(highestChannelSegment)
            messageDao.insert(message)
            message
        }
            .subscribeOn(Schedulers.io())

    }

    fun getMessage(messageId: String): EkoMessageEntity? {
        return UserDatabase.get().messageDao().getByIdNow(messageId)
    }

    fun updateMessageState(messageId: String, state: AmityMessage.State) : Completable {
        return UserDatabase.get().messageDao().updateSyncState(messageId, state.stateName)
            .subscribeOn(Schedulers.io())
    }

    fun observeLatestMessage(channelId: String, isDeleted: Boolean?): Flowable<EkoMessageEntity> {
        return  UserDatabase.get().messageDao().getLatestMessage(channelId, isDeleted)
    }

    fun observeMessage(messageId: String) : Flowable<EkoMessageEntity> {
        return UserDatabase.get().messageDao().getById(messageId)
    }

    fun hardDeleteAllFromChannel(channelId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().messageDao().hardDeleteAllFromChannel(channelId)
        }
    }

    fun hardDeleteMessage(messageId: String): Completable {
        return UserDatabase.get().messageDao().hardDeleteMessage(messageId)
    }

    fun observeMessages(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?
    ): Flowable<List<EkoMessageEntity>> {
        return UserDatabase.get().messageDao().observeMessages(
            channelId,
            isFilterByParentId,
            parentId,
            includingTags,
            excludingTags,
            isDeleted,
            type?.let(::listOf) ?: listOf()
        )
    }

    fun getMessagePagingSource(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        stackFromEnd: Boolean,
        type: String?
    ): PagingSource<Int, EkoMessageEntity> {
        return UserDatabase.get().messagePagingDao().getMessagePagingSource(
            channelId = channelId,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            stackFromEnd = stackFromEnd,
            type = type
        )
    }

    fun getLatestMessage(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<EkoMessageEntity> {
        return UserDatabase.get().messageDao().getLatestMessage(
            channelId,
            isFilterByParentId,
            parentId,
            includingTags.toTypedArray(),
            excludingTags.toTypedArray(),
            isDeleted,
            type?.let(::listOf) ?: listOf(),
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce,
            DateTime.now()
        )
    }

    fun notifyUserUpdate(userId: String) {
        UserDatabase.get().messageDao().updateUser(userId)
    }

}