package com.amity.socialcloud.sdk.chat.data.message

import android.net.Uri
import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.chat.data.message.flag.MessageFlagLocalDataStore
import com.amity.socialcloud.sdk.chat.data.message.flag.MessageFlagRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.message.paging.MessageMediator
import com.amity.socialcloud.sdk.chat.message.AmityMessage
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.core.data.file.FileLocalDataStore
import com.amity.socialcloud.sdk.core.data.file.FileRepository
import com.amity.socialcloud.sdk.core.data.user.UserRemoteDataStore
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.core.file.AmityUploadResult
import com.amity.socialcloud.sdk.core.mention.AmityMentioneeTarget
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.AmityObjectRepository
import com.ekoapp.ekosdk.AmityContentFeedType
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoMessageBoundaryCallback
import com.ekoapp.ekosdk.internal.data.dao.EkoMessageDao
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.comment.CommentLoadResult
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.subjects.PublishSubject

@OptIn(ExperimentalPagingApi::class)
internal class MessageRepository : AmityObjectRepository<EkoMessageEntity, AmityMessage>() {


    override fun fetchAndSave(objectId: String): Completable {
        return MessageRemoteDataStore().getMessage(objectId)
            .flatMapCompletable {
                MessageQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): EkoMessageEntity? {
        val message =  MessageLocalDataStore().getMessage(objectId)
        // We intentionally don't support unsynced object as a live object
        // such as optimistic object (pre-created)
        if (message != null &&
            AmityMessage.State.enumOf(message.syncState) != AmityMessage.State.SYNCED) {
            throw   AmityException.create(
                message = "Observing unsynced object is not supported by Live Object.",
                cause = null,
                error = AmityError.UNSUPPORTED
            )
        }
        return message
    }

    override fun mapper(): ModelMapper<EkoMessageEntity, AmityMessage> {
        return MessageModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EkoMessageEntity> {
        return MessageLocalDataStore().observeMessage(objectId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun createMessage(
        channelId: String,
        parentId: String?,
        fileUri: Uri?,
        type: String,
        data: JsonObject,
        tags: AmityTags,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>
    ): Completable {
        return createPreviewMessage(
            channelId = channelId,
            parentId = parentId,
            type = type,
            fileUri = fileUri,
            data = data,
            tags = tags,
            metadata = metadata,
            mentionees = mentionees
        )
            .flatMapCompletable {
                val messageId = it.messageId
                val dataType = AmityMessage.DataType.enumOf(type)

                val hasUploadState = fileUploadingTypes.contains(dataType)
                if (!hasUploadState) {
                    syncMessage(
                        messageId = messageId,
                        channelId = channelId,
                        parentId = parentId,
                        type = type,
                        data = data,
                        tags = tags,
                        metadata = metadata,
                        mentionees = mentionees,
                    )
                } else {
                    MessageLocalDataStore().updateMessageState(
                        messageId,
                        AmityMessage.State.UPLOADING
                    )
                        .andThen(
                            when (dataType) {
                                AmityMessage.DataType.IMAGE -> {
                                    FileRepository().uploadImage(messageId, fileUri!!)
                                        .flatMapCompletable {
                                            handleUploadResult(
                                                messageId = messageId,
                                                channelId = channelId,
                                                parentId = parentId,
                                                type = type,
                                                data = data,
                                                tags = tags,
                                                metadata = metadata,
                                                mentionees = mentionees,
                                                uploadResult = it
                                            )
                                        }
                                }
                                AmityMessage.DataType.FILE -> {
                                    FileRepository().uploadFile(messageId, fileUri!!)
                                        .flatMapCompletable {
                                            handleUploadResult(
                                                messageId = messageId,
                                                channelId = channelId,
                                                parentId = parentId,
                                                type = type,
                                                data = data,
                                                tags = tags,
                                                metadata = metadata,
                                                mentionees = mentionees,
                                                uploadResult = it
                                            )
                                        }
                                }
                                AmityMessage.DataType.AUDIO -> {
                                    FileRepository().uploadAudio(messageId, fileUri!!)
                                        .flatMapCompletable {
                                            handleUploadResult(
                                                messageId = messageId,
                                                channelId = channelId,
                                                parentId = parentId,
                                                type = type,
                                                data = data,
                                                tags = tags,
                                                metadata = metadata,
                                                mentionees = mentionees,
                                                uploadResult = it
                                            )
                                        }
                                }
                                AmityMessage.DataType.VIDEO -> {
                                    FileRepository().uploadVideo(messageId, fileUri!!, AmityContentFeedType.MESSAGE)
                                            .flatMapCompletable {
                                                handleUploadResult(
                                                        messageId = messageId,
                                                        channelId = channelId,
                                                        parentId = parentId,
                                                        type = type,
                                                        data = data,
                                                        tags = tags,
                                                        metadata = metadata,
                                                        mentionees = mentionees,
                                                        uploadResult = it
                                                )
                                            }
                                }
                                else -> {
                                    // Not a valid case
                                    Completable.complete()
                                }
                            }
                        )
                }
            }
    }

    private fun <T : AmityFileInfo> handleUploadResult(
        messageId: String,
        channelId: String,
        parentId: String?,
        type: String,
        data: JsonObject,
        tags: AmityTags,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>,
        uploadResult: AmityUploadResult<T>
    ): Completable {
        return when (uploadResult) {
            is AmityUploadResult.COMPLETE<T> -> {
                val fileId = uploadResult.getFile().getFileId()
                syncMessage(
                    messageId = messageId,
                    channelId = channelId,
                    parentId = parentId,
                    type = type,
                    fileId = fileId,
                    data = data,
                    tags = tags,
                    metadata = metadata,
                    mentionees = mentionees
                )
            }
            is AmityUploadResult.ERROR -> {
                MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.FAILED)
                    .andThen(Completable.error(uploadResult.getError()))
            }
            is AmityUploadResult.CANCELLED -> {
                MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.FAILED)
                    .andThen(
                        Completable.error(
                            AmityException.create(
                                message = "${type.capitalize()} upload cancelled",
                                null,
                                AmityError.UNKNOWN
                            )
                        )
                    )
            }
            else -> {
                Completable.complete()
            }
        }
    }

    private fun syncMessage(
        messageId: String,
        channelId: String,
        parentId: String?,
        type: String,
        data: JsonObject,
        tags: AmityTags,
        fileId: String? = null,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>
    ): Completable {
        return MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.SYNCING)
            .andThen(MessageRemoteDataStore().createMessage(
                messageId = messageId,
                channelId = channelId,
                parentId = parentId,
                type = type,
                fileId = fileId,
                data = data,
                tags = tags,
                metadata = metadata,
                mentionees = mentionees
            )
                .flatMapCompletable {
                    MessageQueryPersister().persist(it)
                        .andThen(MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.SYNCED))
                })
            .onErrorResumeNext {
                MessageLocalDataStore().updateMessageState(messageId, AmityMessage.State.FAILED)
                    .andThen(Completable.error(it))
            }
    }

    private fun createPreviewMessage(
        channelId: String,
        parentId: String?,
        type: String,
        data: JsonObject?,
        tags: AmityTags,
        fileUri: Uri?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>
    ): Single<EkoMessageEntity> {
        return MessageLocalDataStore().createMessage(
            channelId = channelId,
            parentId = parentId,
            type = type,
            data = data,
            tags = tags,
            metadata = metadata,
            fileUri = fileUri,
            mentionees = mentionees
        ).flatMap { message ->
            MessageFlagLocalDataStore().createFlag(message.messageId)
                .andThen(
                    if (!fileUploadingTypes.contains(message.getDataType())) {
                        Completable.complete()
                    } else {
                        val fileType = when (message.getDataType()) {
                            AmityMessage.DataType.IMAGE -> {
                                "image"
                            }
                            AmityMessage.DataType.VIDEO -> {
                                AmityMessage.DataType.VIDEO.apiKey
                            }
                            AmityMessage.DataType.AUDIO -> {
                                AmityMessage.DataType.AUDIO.apiKey
                            }
                            else -> {
                                "file"
                            }
                        }
                        val path = fileUri?.path ?: ""
                        FileLocalDataStore().createLocalFile(message.messageId, fileType, path)
                    }
                )
                .andThen(Single.just(message))
        }
    }

    fun observeLatestMessage(channelId: String, isDeleted: Boolean?): Flowable<AmityMessage> {
        return MessageLocalDataStore().observeLatestMessage(channelId, isDeleted)
            .map {
                MessageModelMapper().map(it)
            }
    }

    fun updateMessage(
        messageId: String,
        data: JsonObject?,
        tags: AmityTags?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): Completable {
        return MessageRemoteDataStore().updateMessage(messageId, data, tags, metadata, mentionees)
            .map {
                MessageQueryPersister().persist(it)
            }
            .ignoreElement()
    }

    fun deleteMessage(messageId: String): Completable {
        return Single.fromCallable {
            var deletingId = ""
            val message = MessageLocalDataStore().getMessage(messageId)
            if (message == null || message.syncState == AmityMessage.State.SYNCED.stateName) {
                deletingId = messageId
            }
            deletingId
        }
            .flatMapCompletable { deletingId ->
                if (deletingId.isNotEmpty()) {
                    MessageRemoteDataStore().deleteMessage(messageId)
                } else {
                    MessageLocalDataStore().hardDeleteMessage(messageId)
                }
            }
    }

    fun flagMessage(messageId: String): Completable {
        return MessageFlagRemoteDataStore().flagMessage(messageId)
            .map {
                MessageQueryPersister().persist(it)
            }
            .ignoreElement()
    }

    fun unflagMessage(messageId: String): Completable {
        return MessageFlagRemoteDataStore().unflagMessage(messageId)
            .map {
                MessageQueryPersister().persist(it)
            }
            .ignoreElement()
    }

    fun observeMessages(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?
    ): Flowable<List<AmityMessage>> {
        return MessageLocalDataStore().observeMessages(
            channelId,
            isFilterByParentId,
            parentId,
            includingTags,
            excludingTags,
            isDeleted,
            type
        ).map {
            it.map {
                MessageModelMapper().map(it)
            }
        }
    }

    fun loadFirstPageMessages(
        channelId: String,
        stackFromEnd: Boolean,
        parentId: String?,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        type: String?,
        limit: Int
    ): Single<CommentLoadResult> {
        return MessageRemoteDataStore().queryMessages(
            channelId = channelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type,
            skip = 0,
            limit = limit,
            type = "pagination"
        )
            .flatMap {
                MessageLocalDataStore().hardDeleteAllFromChannel(channelId)
                    .andThen(MessageQueryPersister().persist(it))
                    .andThen(Single.just(it))
            }
            .map { dto ->
                val ids = dto.messages.map { it.messageId }
                val token = if (stackFromEnd) dto.token?.previous ?: "" else dto.token?.next ?: ""
                CommentLoadResult(token, ids)
            }
    }

    fun loadMessages(
        channelId: String,
        stackFromEnd: Boolean,
        parentId: String?,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        type: String?,
        token: String
    ): Single<CommentLoadResult> {
        return MessageRemoteDataStore().queryMessages(
            channelId = channelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type,
            token = token,
            type = "pagination"
        )
            .flatMap {
                MessageQueryPersister().persist(it)
                    .andThen(Single.just(it))
            }
            .map { dto ->
                val ids = dto.messages.map { it.messageId }
                val token = if (stackFromEnd) dto.token?.previous ?: "" else dto.token?.next ?: ""
                CommentLoadResult(token, ids)
            }
    }

    fun getMessagePagedList(
        channelId: String,
        stackFromEnd: Boolean,
        parentId: String?,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        type: AmityMessage.DataType?
    ): Flowable<PagedList<AmityMessage>> {

        val userDatabase = UserDatabase.get()
        val messageDao: EkoMessageDao = userDatabase.messageDao()
        val localFactory = messageDao.getDataSource(
            channelId,
            isFilterByParentId,
            parentId,
            includingTags,
            excludingTags,
            isDeleted
        )

        val factory = localFactory
            .map(MessageComposerFunction())

        val delaySubject = PublishSubject.create<Boolean>()

        val boundaryCallback = EkoMessageBoundaryCallback(
            channelId,
            parentId,
            isFilterByParentId,
            isDeleted,
            includingTags,
            excludingTags,
            type,
            stackFromEnd,
            getDefaultPageSize(),
            delaySubject
        )

        return createRxCollectionWithBoundaryCallback(
            factory.map(boundaryCallback),
            boundaryCallback,
            if (stackFromEnd) Int.MAX_VALUE else 0
        ).toV3()
    }

    fun getMessagePagingData(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        stackFromEnd: Boolean,
        type: AmityMessage.DataType?
    ): Flowable<PagingData<AmityMessage>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                prefetchDistance = 0,
                initialLoadSize = (getDefaultPageSize()/2),
            ),
            dynamicQueryStreamMediator = MessageMediator(
                channelId = channelId,
                isFilterByParentId = isFilterByParentId,
                parentId = parentId,
                includingTags = includingTags,
                excludingTags = excludingTags,
                isDeleted = isDeleted,
                stackFromEnd = stackFromEnd,
                type = type?.apiKey
            ),
            pagingSourceFactory = {
                MessageLocalDataStore().getMessagePagingSource(
                    channelId = channelId,
                    isFilterByParentId = isFilterByParentId,
                    parentId = parentId,
                    includingTags = includingTags,
                    excludingTags = excludingTags,
                    isDeleted = isDeleted,
                    stackFromEnd = stackFromEnd,
                    type = type?.apiKey
                )
            },
            modelMapper = MessageModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun getLatestMessage(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<AmityMessage> {
        return MessageLocalDataStore().getLatestMessage(
            channelId = channelId,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            type = type,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                MessageModelMapper().map(it)
            }
    }

    fun isFlaggedByMe(messageId: String): Single<Boolean> {
        return MessageFlagRemoteDataStore().isFlaggedByMe(messageId)
            .map { it.get("isFlagByMe").asBoolean }
    }

    companion object {
        private val fileUploadingTypes = listOf(
            AmityMessage.DataType.IMAGE,
            AmityMessage.DataType.FILE,
            AmityMessage.DataType.AUDIO,
            AmityMessage.DataType.VIDEO
        )
    }
}
