package com.amity.socialcloud.sdk.chat.data.message.paging

import com.amity.socialcloud.sdk.chat.data.message.MessageQueryPersister
import com.amity.socialcloud.sdk.chat.data.message.MessageRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.message.MessageRepository
import com.amity.socialcloud.sdk.chat.message.AmityMessage
import com.amity.socialcloud.sdk.core.AmityTags
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

internal class MessageMediator(
    private val channelId: String,
    private val isFilterByParentId: Boolean,
    private val parentId: String?,
    private val includingTags: AmityTags,
    private val excludingTags: AmityTags,
    private val isDeleted: Boolean?,
    private val stackFromEnd: Boolean,
    private val type: String?
) : DynamicQueryStreamMediator<EkoMessageEntity, MessageQueryDto, AmityMessage>(

    nonce = AmityNonce.MESSAGE_LIST,
    dynamicQueryStreamKeyCreator = MessageKeyCreator(
        channelId = channelId,
        parentId = parentId,
        isFilterByParentId = isFilterByParentId,
        includingTags = includingTags,
        excludingTags = excludingTags,
        isDeleted = isDeleted,
        stackFromEnd = stackFromEnd,
        type = type
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityMessage> {
        return MessageRepository().getLatestMessage(
            channelId = channelId,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            type = type,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toV2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<MessageQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<MessageQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: MessageQueryDto): Completable {
        return MessageQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: MessageQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.previous,
            previous = dto.token?.next,
            primaryKeys = dto.messages?.map { it.messageId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<MessageQueryDto> {
        return MessageRemoteDataStore().queryMessages(
            channelId = channelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type,
            limit = limit,
            last = limit.takeIf { token == null},
            token = token
        ).toV2()
    }

    override fun stackFromEnd(): Boolean {
        return false
    }

}