package com.amity.socialcloud.sdk.core.data.follow

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.follow.info.MyFollowInfoModelMapper
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.user.*
import com.amity.socialcloud.sdk.social.data.follow.FollowRelationshipModelMapper
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.AmityObjectRepository
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.FollowersBoundaryCallback
import com.ekoapp.ekosdk.internal.data.boundarycallback.FollowingsBoundaryCallback
import com.ekoapp.ekosdk.internal.data.model.EkoFollowCountEntity
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.user.FollowerMediator
import com.ekoapp.ekosdk.internal.repository.user.FollowingMediator
import com.ekoapp.ekosdk.internal.repository.user.helper.MapToUserFollowStatusModelHelper
import com.ekoapp.ekosdk.internal.repository.user.helper.UserRepositoryHelper
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class FollowRepository : AmityObjectRepository<EkoFollowCountEntity, AmityMyFollowInfo>() {

    override fun fetchAndSave(objectId: String): Completable {
        return FollowRemoteDataStore().getMyFollowInfo()
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): EkoFollowCountEntity? {
        return FollowLocalDataStore().getFollowCount(objectId)
    }

    override fun mapper(): ModelMapper<EkoFollowCountEntity, AmityMyFollowInfo> {
        return MyFollowInfoModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EkoFollowCountEntity> {
        return FollowLocalDataStore().observeFollowCount(objectId)
    }

    fun getUserFollowInfo(userId: String): Flowable<AmityUserFollowInfo> {
        val myUserId = AmityCoreClient.getUserId()

        return FollowRemoteDataStore().getUserFollowInfo(userId)
                .flatMapCompletable {
                    FollowQueryPersister().persist(it)
                }.andThen(
                    Flowable.combineLatest(
                            FollowLocalDataStore().observeUserFollow(myUserId, userId),
                            FollowLocalDataStore().observeFollowCount(userId)
                    ) { followEntity: EkoUserFollowStatusEntity, followCountEntity: EkoFollowCountEntity ->
                        val mapFollow = followEntity.let(UserRepositoryHelper()::attachDataAndMapToExternal)
                        UserRepositoryHelper().attachUserFollowInfoDataAndMapToExternal(
                                mapFollow,
                                followCountEntity
                        )
                    }
                )
    }

    fun requestFollow(userId: String): Single<AmityFollowStatus> {
        return FollowRemoteDataStore().requestFollow(userId)
            .map {
                FollowQueryPersister().persist(it)
                val status = it.follows.first().status
                AmityFollowStatus.enumOf(status)
            }
    }

    fun acceptFollow(userId: String): Completable {
        return Completable.fromAction {
            FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.ACCEPTED)
        }.concatWith(
            FollowRemoteDataStore().acceptFollow(userId)
                .flatMapCompletable {
                    //TODO Remove after backend return follow count with accept payload
                    val followLocalDataStore = FollowLocalDataStore()
                    val myUserId = AmityCoreClient.getUserId()
                    val followCountEntity = followLocalDataStore.getFollowCountByIdNow(myUserId)
                    if (followCountEntity.followerCount != null && followCountEntity.followerCount!! >= 0) {
                        followLocalDataStore.updateFollowerCount(
                            myUserId,
                            followCountEntity.followerCount!!.inc()
                        )
                    }
                    if (followCountEntity.pendingCount != null && followCountEntity.pendingCount!! >= 0) {
                        followLocalDataStore.updatePendingCount(
                            myUserId,
                            followCountEntity.pendingCount!!.dec()
                        )
                    }
                    Completable.complete()
                }
                .doOnError {
                    if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                        FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
                    } else {
                        FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.PENDING)
                    }
                })
    }

    fun declineFollow(userId: String): Completable {
        return Completable.fromAction {
            FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
        }.concatWith(
            FollowRemoteDataStore().declineFollow(userId)
                .flatMapCompletable {
                    //TODO Remove after backend return follow count with accept payload
                    val followLocalDataStore = FollowLocalDataStore()
                    val myUserId = AmityCoreClient.getUserId()
                    val followCountEntity = followLocalDataStore.getFollowCountByIdNow(myUserId)
                    if (followCountEntity.pendingCount != null && followCountEntity.pendingCount!! >= 0) {
                        followLocalDataStore.updatePendingCount(
                            myUserId,
                            followCountEntity.pendingCount!!.dec()
                        )
                    }
                    Completable.complete()
                }
                .doOnError {
                    if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                        FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
                    } else {
                        FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.PENDING)
                    }
                })
    }

    fun removeFollower(userId: String): Completable {
        return Completable.fromAction {
            FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
        }.concatWith(
            FollowRemoteDataStore().declineFollow(userId)
                .flatMapCompletable {
                    val followLocalDataStore = FollowLocalDataStore()
                    val myUserId = AmityCoreClient.getUserId()
                    val followCountEntity = followLocalDataStore.getFollowCountByIdNow(myUserId)
                    if (followCountEntity.followerCount != null && followCountEntity.followerCount!! >= 0) {
                        followLocalDataStore.updateFollowerCount(
                            myUserId,
                            followCountEntity.followerCount!!.dec()
                        )
                    }
                    Completable.complete()
                }
                .doOnError {
                    if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                        FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
                    } else {
                        FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.ACCEPTED)
                    }
                })
    }

    fun unfollow(userId: String): Completable {
        return Completable.fromAction {
            FollowLocalDataStore().updateFollowing(userId, AmityFollowStatus.NONE)
        }.concatWith(
            FollowRemoteDataStore().unfollow(userId)
                .flatMapCompletable {
                    val followLocalDataStore = FollowLocalDataStore()
                    val followCountEntity = followLocalDataStore.getFollowCountByIdNow(userId)
                    if (followCountEntity.followingCount != null && followCountEntity.followingCount!! >= 0) {
                        followLocalDataStore.updateFollowingCount(
                            userId,
                            followCountEntity.followingCount!!.dec()
                        )
                    }
                    return@flatMapCompletable Completable.complete()
                }
                .doOnError {
                    if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                        FollowLocalDataStore().updateFollowing(userId, AmityFollowStatus.NONE)
                    } else {
                        FollowLocalDataStore().updateFollowing(userId, AmityFollowStatus.ACCEPTED)
                    }
                })
    }

    fun getFollowingsPagedList(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<PagedList<AmityFollowRelationship>> {
        val followDao = UserDatabase.get().followDao()
        val factory = followDao.getFollowingsDataSource(userId, filter)
        val boundaryCallback = FollowingsBoundaryCallback(
            userId,
            filter,
            getDefaultPageSize()
        )
        return createRxCollectionWithBoundaryCallback(
            factory.map(MapToUserFollowStatusModelHelper()),
            boundaryCallback
        ).toV3()
    }

    fun getFollowingsPagingData(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<PagingData<AmityFollowRelationship>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = FollowingMediator(
                userId = userId,
                filter = filter
            ),
            pagingSourceFactory = {
                FollowLocalDataStore().getFollowingsPagingSource(
                    userId = userId,
                    filter = filter
                )
            },
            modelMapper = FollowRelationshipModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun getFollowersPagedList(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<PagedList<AmityFollowRelationship>> {
        val followerDao = UserDatabase.get().followDao()
        val factory = followerDao.getFollowersDataSource(userId, filter)
        val boundaryCallback = FollowersBoundaryCallback(
            userId,
            filter,
            getDefaultPageSize()
        )
        return createRxCollectionWithBoundaryCallback(
            factory.map(MapToUserFollowStatusModelHelper()),
            boundaryCallback
        ).toV3()
    }

    fun getFollowersPagingData(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<PagingData<AmityFollowRelationship>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = FollowerMediator(
                userId = userId,
                filter = filter
            ),
            pagingSourceFactory = {
                FollowLocalDataStore().getFollowersPagingSource(
                    userId = userId,
                    filter = filter
                )
            },
            modelMapper = FollowRelationshipModelMapper()
        )

        return pagerCreator.create().toV3()
    }

    fun getLatestFollowing(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<AmityFollowRelationship> {
        return FollowLocalDataStore().getLatestFollowing(userId = userId, filter = filter)
            .map { FollowRelationshipModelMapper().map(it) }
    }

    fun getLatestFollower(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<AmityFollowRelationship> {
        return FollowLocalDataStore().getLatestFollower(userId = userId, filter = filter)
            .map { FollowRelationshipModelMapper().map(it) }
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

}