package com.amity.socialcloud.sdk.core.data.reaction

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.core.data.reaction.paging.ReactionMediator
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.UserLocalDataStore
import com.amity.socialcloud.sdk.core.reaction.AmityReaction
import com.amity.socialcloud.sdk.core.reaction.ReactionReferenceType
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.AmityReactionBoundaryCallback
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Maybe
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.subjects.PublishSubject

internal class ReactionRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun getReactionCollection(
        referenceType: ReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): Flowable<PagedList<AmityReaction>> {
        val reactionDao = UserDatabase.get().reactionDao()
        val localFactory = if (reactionName.isNullOrEmpty()) {
            reactionDao.getAllByReferenceId(referenceId, referenceType.value)
        } else {
            reactionDao.getAllByReferenceIdAndReactionName(
                referenceId,
                referenceType.value,
                reactionName
            )
        }
        val factory = localFactory.map(ReactionComposerFunction())
        val delaySubject = PublishSubject.create<Boolean>()
        val boundaryCallback = AmityReactionBoundaryCallback(
            referenceId = referenceId,
            referenceType = referenceType,
            pageSize = getDefaultPageSize(),
            delaySubject = delaySubject
        )
        return createRxCollectionWithBoundaryCallback(
            factory.map(boundaryCallback),
            boundaryCallback
        ).toV3()
    }

    @OptIn(ExperimentalPagingApi::class)
    fun getReactionPagingData(
        referenceType: ReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): Flowable<PagingData<AmityReaction>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = ReactionMediator(
                referenceType = referenceType,
                referenceId = referenceId,
                reactionName = reactionName
            ),
            pagingSourceFactory = {
                ReactionLocalDataStore().getReactionsPagingSource(
                    referenceType = referenceType,
                    referenceId = referenceId,
                    reactionName = reactionName
                )
            },
            modelMapper = ReactionModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun getMyReactions(referenceType: ReactionReferenceType, referenceId: String): List<String> {
        var list = emptyList<String>()
        Completable.fromCallable {
            val reactionDao = UserDatabase.get().reactionDao()
            val entities = reactionDao.getAllMyReactionsByReferenceTypeAndReferenceIdNow(
                referenceType.value,
                referenceId
            )
            if (entities.isNotEmpty()) {
                list = entities.map {
                    it.reactionName
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return list
    }

    fun getLatestReaction(
        referenceType: ReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): Flowable<AmityReaction> {
        return ReactionLocalDataStore().getLatestReaction(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName
        ).map {
            ReactionModelMapper().map(it)
        }
    }

    fun addReaction(referenceType: ReactionReferenceType, referenceId: String, reactionName: String) : Completable {
        return getMyReaction(referenceType, referenceId, reactionName)
            .subscribeOn(Schedulers.io())
            .isEmpty
            .flatMapCompletable { isEmpty: Boolean ->
                if (isEmpty) {
                    val reaction = createMyReaction(referenceType, referenceId, reactionName)
                    ReactionLocalDataStore().addReaction(reaction)
                        .concatWith(
                            ReactionRemoteDataStore().addReaction(referenceType, referenceId, reactionName)
                                .ignoreElement()
                                .onErrorResumeNext {
                                    // revert local value
                                    ReactionLocalDataStore().removeReaction(reaction)
                                }
                        )
                } else {
                    Completable.complete()
                }
            }
    }

    fun removeReaction(referenceType: ReactionReferenceType, referenceId: String, reactionName: String) : Completable {
        return getMyReaction(referenceType, referenceId, reactionName)
            .subscribeOn(Schedulers.io())
            .flatMapCompletable { removingReaction: EkoReactionEntity ->
                ReactionLocalDataStore().removeReaction(removingReaction)
                    .concatWith(
                        ReactionRemoteDataStore().removeReaction(referenceType, referenceId, reactionName)
                            .ignoreElement()
                            .onErrorResumeNext {
                                // revert local value
                                ReactionLocalDataStore().addReaction(removingReaction)
                            }
                    )
            }
    }

    private fun getMyReaction(referenceType: ReactionReferenceType, referenceId: String, reactionName: String) : Maybe<EkoReactionEntity> {
        return ReactionLocalDataStore().getReaction(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName,
            userId = SessionLocalDataStore().getActiveUserId()
        )
    }

    fun createMyReaction(referenceType: ReactionReferenceType, referenceId: String, reactionName: String) : EkoReactionEntity {
        return ReactionLocalDataStore().createReaction(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName,
            userId =  SessionLocalDataStore().getActiveUserId()
        )
    }

}