package com.amity.socialcloud.sdk.core.data.session

import android.util.Log
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.data.notification.device.DeviceNotificationRepository
import com.amity.socialcloud.sdk.core.data.user.UserQueryPersister
import com.amity.socialcloud.sdk.core.data.user.UserRepository
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.exception.EntityNotFoundException
import com.ekoapp.core.utils.toV2
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.dto.EkoUserListDto
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.internal.repository.user.UserUpdateOption
import com.github.davidmoten.rx2.RetryWhen
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import org.joda.time.Duration
import java.util.concurrent.TimeUnit

internal class SessionRepository : EkoObjectRepository() {

    private val ABOUT_TO_EXPIRE_TRESHOLD = 0.8

    fun getActiveUserId(): String {
        return SessionLocalDataStore().getActiveUserId()
    }

    fun login(
        userId: String,
        displayName: String?,
        authToken: String?,
        isLegacyVersion: Boolean
    ): Completable {
        return activateAccount(userId)
            .flatMapCompletable {
                renewTokenIfNeed(it, displayName, authToken, isLegacyVersion)
            }
            .doOnComplete {
                // Do this here; to not break the stream when there is an error
                updateDisplayNameIfNeeded(userId, displayName)
            }
    }

    fun logout(): Completable {
        return DeviceNotificationRepository().unregisterAll()
    }

    fun clearData(): Completable {
        return Completable.fromAction {
            SessionLocalDataStore().logoutAccount(SessionLocalDataStore().getActiveUserId())
        }.subscribeOn(Schedulers.io())
    }

    fun getCurrentAccount(): Single<EkoAccount> {
        return Single.fromCallable {
            val account = SessionLocalDataStore().getCurrentAccount()
            if (account != null) {
                return@fromCallable account
            } else {
                throw EntityNotFoundException
            }
        }.subscribeOn(Schedulers.io())
    }


    fun renewToken(
        account: EkoAccount,
        displayName: String? = null,
        authToken: String?,
        isLegacyVersion: Boolean
    ): Completable {
        return SessionLocalDataStore().getApiKey()
            .flatMapCompletable {
                SessionRemoteDataStore().registerDevice(
                    apiKey = it.apiKey,
                    userId = account.userId,
                    displayName = displayName,
                    deviceId = account.deviceId,
                    authToken = authToken,
                    isLegacyVersion = isLegacyVersion
                )
                    .flatMapCompletable {
                        account.refreshToken = it.refreshToken
                        account.accessToken = it.accessToken
                        if (isLegacyVersion) {
                            //for legacy version, there's no expiration yet
                            account.issuedAt =  DateTime.now()
                            account.expiresAt = DateTime.now().plus(Duration.standardDays(365))
                        } else {
                            account.issuedAt =  it.issuedAt
                            account.expiresAt = it.expiresAt
                        }
                        val tokenDuration = (account.expiresAt.millis
                                - account.issuedAt.millis
                                - AmityCoreClient.millisTimeDiff) * ABOUT_TO_EXPIRE_TRESHOLD
                        account.aboutToExpireAt = account.issuedAt.plus(tokenDuration.toLong())


                        Log.e("SSM3", "token renewed: account updated : \n " +
                                "      expiresAt = ${account.expiresAt} \naboutToExpireAt = ${account.aboutToExpireAt}\n" +
                                "       issuedAt = ${account.issuedAt}" +
                                "\nduration: = ${tokenDuration/1000} sec ")

                        val users = EkoUserListDto().apply {
                            users = it.users ?: emptyList()
                            files = it.files ?: emptyList()
                        }
                        SessionLocalDataStore().updateAccount(account)
                            .andThen(UserQueryPersister().persist(users))
                    }
                    .toV2()
                    .retryWhen(
                        RetryWhen
                            .retryIf { AmityError.from(it) != AmityError.USER_IS_GLOBAL_BANNED }
                            .maxRetries(3)
                            .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                            .build()
                    ).doOnError {
                        SessionLocalDataStore().logoutAccount(account.userId)
                    }
                    .toV3()
            }
    }

    private fun activateAccount(userId: String): Single<EkoAccount> {
        return Single.fromCallable {
            SessionLocalDataStore().activateAccount(userId)
        }.subscribeOn(Schedulers.io())
    }

    private fun renewTokenIfNeed(
        account: EkoAccount,
        displayName: String?,
        authToken: String?,
        isLegacyVersion: Boolean
    ): Completable {
        return verifyCurrentAccessToken(account).onErrorResumeNext {
                if (AmityError.from(it) == AmityError.USER_IS_GLOBAL_BANNED) {
                    Completable.error(it)
                } else {
                    renewToken(account, displayName, authToken, isLegacyVersion)
                }
            }
    }

    private fun verifyCurrentAccessToken(account: EkoAccount): Completable {
        return SessionLocalDataStore().getApiKey()
            .flatMapCompletable {
                if (account.accessToken == null) {
                    Completable.error(
                        AmityException.create(
                            "accessToken not found",
                            null,
                            AmityError.UNAUTHORIZED_ERROR
                        )
                    )
                } else {
                    SessionRemoteDataStore().verifyAccessToken(it.apiKey)
                }
            }
    }

    private fun updateDisplayNameIfNeeded(userId: String, displayName: String?) {
        if (displayName != null) {
            UserRepository().observe(userId)
                .firstOrError()
                .flatMapCompletable {
                    if (it.getDisplayName() == displayName) {
                        Completable.complete()
                    } else {
                        UserRepository().updateUser(
                            userId,
                            UserUpdateOption(displayName = displayName)
                        )
                            .ignoreElement()
                    }
                }
                .subscribeOn(Schedulers.io())
                .doOnError {
                    // do nothing
                }
                .subscribe()
        }
    }

}