package com.amity.socialcloud.sdk.core.data.user

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipLocalDataStore
import com.amity.socialcloud.sdk.chat.data.message.MessageLocalDataStore
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.user.paging.UserMediator
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.amity.socialcloud.sdk.core.user.AmityUserSortOption
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipLocalDataStore
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.AmityObjectRepository
import com.ekoapp.ekosdk.UserEntity
import com.ekoapp.ekosdk.internal.api.socket.request.UserUpdateRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.UserBoundaryCallback
import com.ekoapp.ekosdk.internal.data.dao.UserDao
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.user.UserUpdateOption
import com.ekoapp.ekosdk.internal.repository.user.helper.UserComposerFunction
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class UserRepository : AmityObjectRepository<UserEntity, AmityUser>() {

    override fun fetchAndSave(objectId: String): Completable {
        return UserRemoteDataStore().getUser(objectId)
            .flatMap {
                UserQueryPersister().persist(it)
                    .andThen(Single.just((it.users)))
            }
            .doOnSuccess {
                if (it.isNullOrEmpty()) {
                    throw AmityException.create(
                        message = AmityError.ITEM_NOT_FOUND.name,
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                }
            }
            .ignoreElement()
    }

    override fun queryFromCache(objectId: String): UserEntity? {
        return  UserLocalDataStore().getUser(objectId)
    }

    override fun mapper(): ModelMapper<UserEntity, AmityUser> {
        return UserModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<UserEntity> {
        return UserLocalDataStore().observeUser(objectId)
    }

    @Deprecated("Use getUserPagingData instead")
    fun getUserPagedList(
        keyword: String,
        sortBy: AmityUserSortOption
    ): io.reactivex.Flowable<PagedList<AmityUser>> {
        val userDao: UserDao = UserDatabase.get().userDao()
        val factory = userDao.getDataSource(keyword, sortBy)
        val boundaryCallback = UserBoundaryCallback(
            keyword,
            sortBy.apiKey,
            getDefaultPageSize()
        )

        return createRxCollectionWithBoundaryCallback(
            factory.map(UserComposerFunction()),
            boundaryCallback
        )
    }

    fun getUserPagingData(
        keyword: String?,
        sortBy: AmityUserSortOption
    ): Flowable<PagingData<AmityUser>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = UserMediator(
                keyword = keyword,
                sortBy = sortBy
            ),
            pagingSourceFactory = {
                UserLocalDataStore().getUserPagingSource(
                    keyword = keyword,
                    sortBy = sortBy
                )
            },
            modelMapper = UserModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun getUser(userId: String): AmityUser? {
        return UserLocalDataStore().getUser(userId)?.let {
            UserModelMapper().map(it)
        }
    }

    fun updateUser(userId: String, option: UserUpdateOption): Single<AmityUser> {
        val request = UserUpdateRequest(
            userId = userId,
            displayName = option.displayName,
            roles = option.roles,
            metadata = option.metadata,
            avatarFileId = option.avatarFileId,
            avatarCustomUrl = option.avatarCustomUrl,
            description = option.description
        )
        return UserRemoteDataStore().updateUser(request)
            .flatMap {
                UserQueryPersister().persist(it)
                    .andThen(Single.just((it.users.first().userId)))
            }.map {
                getUser(it)
            }
    }

    fun hasInLocal(userId: String): Boolean {
        return UserLocalDataStore().getUser(userId) != null
    }

    fun flagUser(userId: String): Completable {
        return UserRemoteDataStore().flagUser(userId)
            .flatMapCompletable {
                UserQueryPersister().persist(it)
                    .andThen(notifyUserUpdate(userId))
            }
    }

    fun unflagUser(userId: String): Completable {
        return UserRemoteDataStore().unflagUser(userId)
            .flatMapCompletable {
                UserQueryPersister().persist(it)
                    .andThen(notifyUserUpdate(userId))
            }
    }

    fun isFlaggedByMe(userId: String): Single<Boolean> {
        return UserRemoteDataStore().isFlaggedByMe(userId)
            .map { it.get("isFlagByMe").asBoolean }
    }

    //  dummy update, for triggering user update tied to related entity
    private fun notifyUserUpdate(userId: String): Completable {
        return Completable.fromAction {
            MessageLocalDataStore().notifyUserUpdate(userId)
            ChannelMembershipLocalDataStore().notifyUserUpdate(userId)
            CommunityMembershipLocalDataStore().notifyUserUpdate(userId)
        }
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }
}