package com.amity.socialcloud.sdk.core.events

import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.domain.topic.*
import com.amity.socialcloud.sdk.core.events.comment.AmityCommentEvents
import com.amity.socialcloud.sdk.core.events.community.AmityCommunityEvents
import com.amity.socialcloud.sdk.core.events.post.AmityPostEvents
import com.amity.socialcloud.sdk.core.events.relationship.AmityFollowEvents
import com.amity.socialcloud.sdk.core.events.user.AmityUserEvents
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.amity.socialcloud.sdk.social.comment.AmityComment
import com.amity.socialcloud.sdk.social.community.AmityCommunity
import com.amity.socialcloud.sdk.social.feed.AmityPost
import com.ekoapp.core.utils.toV2
import io.reactivex.Single

sealed class AmityTopic(
    private val nonce: String,
    private val id: String,
    private val events: String
) {

    private fun getKey(): String {
        return nonce + id + events
    }

    internal abstract fun generateTopic(): Single<String>

    override fun equals(other: Any?): Boolean {
        return other is AmityTopic && other.getKey() == getKey()
    }

    override fun hashCode(): Int {
        return getKey().hashCode()
    }

    class COMMUNITY(val community: AmityCommunity, val events: AmityCommunityEvents) :
        AmityTopic("community", community.getCommunityId(), events.name) {

        override fun generateTopic(): Single<String> {
            return CommunityTopicGeneratorUseCase().execute(community, events)
        }

    }

    class COMMENT(val comment: AmityComment, val events: AmityCommentEvents) :
        AmityTopic("comment", comment.getCommentId(), events.name) {

        override fun generateTopic(): Single<String> {
            return CommentTopicGeneratorUseCase().execute(comment, events)
        }

    }

    class USER(val user: AmityUser, val events: AmityUserEvents) :
        AmityTopic("user", user.getUserId(), events.name) {

        override fun generateTopic(): Single<String> {
            return UserTopicGeneratorUseCase().execute(user, events)
        }

    }

    class POST(val post: AmityPost, val events: AmityPostEvents) :
        AmityTopic("post", post.getPostId(), events.name) {

        override fun generateTopic(): Single<String> {
            return PostTopicGeneratorUseCase().execute(post, events)
        }
    }

    class FOLLOW(val events: AmityFollowEvents) :
        AmityTopic("follow", AmityCoreClient.getUserId(), events.name) {

        override fun generateTopic(): Single<String> {
            return FollowTopicGeneratorUseCase().execute(events).toV2()
        }
    }

    internal class NETWORK
        : AmityTopic("network", AmityCoreClient.getUserId(), "") {

        override fun generateTopic(): Single<String> {
            return NetworkTopicGeneratorUseCase().execute().toV2()
        }
    }

}