package com.amity.socialcloud.sdk.core.session

import android.util.Log
import com.amity.socialcloud.sdk.core.domain.session.GetCurrentAccountUseCase
import com.amity.socialcloud.sdk.core.exception.EntityNotFoundException
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionLifeCycle
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.core.session.transformer.SessionLifeCycleTransformer
import com.amity.socialcloud.sdk.core.session.transformer.SessionStateTransformer
import io.reactivex.rxjava3.schedulers.Schedulers

class SessionStateManager(
    private val appEventBus: AppEventBus,
    private val sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    private val sessionStateEventBus: SessionStateEventBus
) {

    init {
        observeAppEvent()
        observeSessionState()
        verifyAccount()
        thisIsTestingLog()
    }

    var sessionState: SessionState = SessionState.NotLoggedIn

    // observe app events then transform and publish to session state
    private fun observeAppEvent() {
        this.appEventBus.observe().doOnNext { appEvent ->
            SessionStateTransformer.transform(
                currentSessionState = this.sessionState,
                appEvent = appEvent
            )?.let { sessionState ->
                this.sessionState = sessionState
                this.sessionStateEventBus.publish(this.sessionState)
            }
        }.subscribe()
    }

    // observe app events then transform and publish to session life cycle
    private fun observeSessionState() {
        this.sessionStateEventBus.observe().doOnNext { sessionState ->
            this.sessionState = sessionState
            Log.e("SSM3", "sessionStateEventBus: ${sessionState}")
            SessionLifeCycleTransformer.transform(sessionState)?.let { sessionLifeCycle ->
                sessionLifeCycleEventBus.publish(sessionLifeCycle)
            }
        }.subscribe()
    }

    private fun verifyAccount() {
        GetCurrentAccountUseCase().execute()
            .subscribeOn(Schedulers.io())
            .doOnSuccess {
                if (!it.accessToken.isNullOrEmpty()) {
                    appEventBus.publish(AppEvent.AppLaunchWithUser)
                    sessionLifeCycleEventBus.publish(SessionLifeCycle.Establish(it))
                } else {
                    appEventBus.publish(AppEvent.AppLaunchWithoutUser)
                }
            }
            .doOnError {
                if (it is EntityNotFoundException) {
                    appEventBus.publish(AppEvent.AppLaunchWithoutUser)
                }
            }
            .subscribe()
    }

    // FIXME: to remove - for testing only
    private fun thisIsTestingLog() {
        sessionLifeCycleEventBus.observe().doOnNext {
            Log.e("SSM3", "sessionLifeCycleEventBus: ${it}")
        }.subscribe()

        appEventBus.observe().doOnNext {
            Log.e("SSM3", "appEventBus: ${it}")
        }.subscribe()
    }
}