package com.amity.socialcloud.sdk.core.session.component

import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.session.AccessTokenRenewalImpl
import com.amity.socialcloud.sdk.core.session.SessionHandler
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import org.joda.time.DateTime
import org.joda.time.Duration

class TokenRenewalSessionComponent(
    private val appEventBus: AppEventBus,
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(
    sessionLifeCycleEventBus, sessionStateEventBus
) {

    var sessionHandler: SessionHandler? = null
    private var currentRenewal: AccessTokenRenewalImpl? = null

    //initiate the first one to be never failed
    private var lastFailedDateTime = DateTime.now().minus(Duration.standardDays(999))

    init {
        observeTokenEvent()
    }

    override fun onSessionStateChange(sessionState: SessionState) {
    }

    override fun establish(account: EkoAccount) {
    }

    override fun destroy() {
    }

    override fun handleTokenExpire() {
        finishCurrentRenewal()
    }

    private fun observeTokenEvent() {
        appEventBus.observe()
            .filter { it is AppEvent.TokenExpire || it is AppEvent.TokenAboutToExpire }
            .doOnNext { initiateRenewalIfNeeded() }
            .subscribe()
    }

    private fun initiateRenewalIfNeeded() {
        if (!canInitiateRenewal()) {
            return
        }
        val accessTokenRenewal = AccessTokenRenewalImpl(
            onRenewTokenSuccess = {
                appEventBus.publish(AppEvent.TokenRenewSuccess)
                finishCurrentRenewal()
            },
            onRenewTokenFailed = { throwable ->
                val amityError = AmityException.fromThrowable(throwable)
                if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                    amityError.code == AmityError.UNAUTHORIZED_ERROR.code
                ) {
                    appEventBus.publish(AppEvent.TerminationCodeReceive(amityError))
                    finishCurrentRenewal()
                } else {
                    unableToRenewWithCurrentRenewal()
                }
            },
            onUnableToRetrieveAuthToken = {
                unableToRenewWithCurrentRenewal()
            },
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = getSessionLifeCycleEventBus()
        )
        initiateRenewal(accessTokenRenewal)
        sessionHandler?.sessionWillRenewAccessToken(accessTokenRenewal)
    }

    private fun initiateRenewal(renewal: AccessTokenRenewalImpl) {
        this.currentRenewal = renewal
    }

    private fun canInitiateRenewal(): Boolean {
        if (this.currentRenewal != null) {
            // Already initiated access token renewal flow.
            // The SDK is waiting for user to give the renewal response.
            return false
        }
        val tenMinutes = Duration.standardMinutes(10)
        if (this.currentRenewal != null
            && DateTime.now().isAfter(this.lastFailedDateTime.plus(tenMinutes))
        ) {
            // The new renewal can be triggered only after 10 minutes from last fail.
            return false
        }
        return true
    }

    private fun unableToRenewWithCurrentRenewal() {
        this.currentRenewal?.invalidate()
        this.currentRenewal = null
        this.lastFailedDateTime = DateTime.now()
    }

    private fun finishCurrentRenewal() {
        this.currentRenewal?.invalidate()
        this.currentRenewal = null
        this.lastFailedDateTime = DateTime.now().minus(Duration.standardDays(999))
    }
}