package com.amity.socialcloud.sdk.infra

import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.AmityEndpoint
import com.amity.socialcloud.sdk.core.encryption.AmityDBEncryption
import com.amity.socialcloud.sdk.core.session.SessionStateManager
import com.amity.socialcloud.sdk.core.session.component.DatabaseSessionComponent
import com.amity.socialcloud.sdk.core.session.component.TokenRenewalSessionComponent
import com.amity.socialcloud.sdk.core.session.component.TokenWatcherSessionComponent
import com.amity.socialcloud.sdk.core.session.component.UserSettingSessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.infra.mqtt.AmityMqttClient
import com.amity.socialcloud.sdk.log.AmityLog
import com.ekoapp.ekosdk.internal.api.EkoApi
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoApiKey
import com.ekoapp.ekosdk.internal.data.model.EkoHttpUrl
import com.ekoapp.ekosdk.internal.data.model.EkoMqttUrl
import com.ekoapp.ekosdk.internal.data.model.EkoSocketUrl
import com.ekoapp.ekosdk.internal.init.AmityCoreSDKInitializer
import com.ekoapp.ekosdk.internal.init.EkoSdkInitProvider
import com.ekoapp.ekosdk.internal.util.AppContext
import com.ekoapp.ekosdk.internal.util.EkoPreconditions
import com.google.common.base.Objects
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.CompletableSubject

internal object CoreClient {

    private val TAG = CoreClient::class.java.name

    internal var mqttClient: AmityMqttClient? = null
    internal var socketClient: EkoSocket? = null
    internal var renewalManager: TokenRenewalSessionComponent? = null

    internal var sessionLifeCycleEventBus: SessionLifeCycleEventBus? = null
    internal var appEventBus: AppEventBus? = null
    internal var sessionStateEventBus: SessionStateEventBus? = null
    internal var sessionStateManager: SessionStateManager? = null

    fun setup(
        apiKey: String,
        endpoint: AmityEndpoint = AmityEndpoint.SG,
        dbEncryption: AmityDBEncryption = AmityDBEncryption.NONE
    ): Completable {
        if (!EkoSdkInitProvider.isInitialized()) {
            return Completable.never()
        }
        EkoPreconditions.checkValidId(apiKey, "apiKey")

        EkoDatabase.init(AppContext.get(), dbEncryption)
        UserDatabase.init(AppContext.get(), dbEncryption)

        val result = CompletableSubject.create()
        val database = EkoDatabase.get()
        val httpUrlDao = database.httpUrlDao()
        val socketUrlDao = database.socketUrlDao()
        val mqttUrlDao = database.mqttUrlDao()
        val newHttpUrl = endpoint.httpEndpoint
        val newSocketUrl = endpoint.socketEndpoint
        val newMqttUrl = endpoint.mqttEndpoint
        val apiKeyDao = EkoDatabase.get().apiKeyDao()

        Maybe.zip(
            apiKeyDao.currentApiKey,
            httpUrlDao.currentHttpUrl,
            socketUrlDao.currentSocketUrl,
            mqttUrlDao.currentMqttUrl
        ) { apiKey: EkoApiKey, httpEntity: EkoHttpUrl, socketEntity: EkoSocketUrl, mqttEntity: EkoMqttUrl ->
            val storedEndpoint =
                AmityEndpoint.CUSTOM(httpEntity.httpUrl, socketEntity.socketUrl, mqttEntity.mqttUrl)
            Pair(apiKey, storedEndpoint)
        }.subscribeOn(Schedulers.io())
            .doOnSuccess { setupPair: Pair<EkoApiKey, AmityEndpoint> ->
                val storedApiKey = setupPair.first.apiKey
                val storedHttpUrl = setupPair.second.httpEndpoint
                val storedSocketUrl = setupPair.second.socketEndpoint
                val storedMqttUrl = setupPair.second.mqttEndpoint
                if (!Objects.equal(storedHttpUrl, newHttpUrl)
                    || !Objects.equal(storedSocketUrl, newSocketUrl)
                    || !Objects.equal(storedMqttUrl, newMqttUrl)
                    || !Objects.equal(storedApiKey, apiKey)
                ) {
                    AmityLog.tag(TAG).e("Setup value changed. new api key: %s new http url: %s new socket url: %s new mqtt url: %s",
                        apiKey,
                        newHttpUrl,
                        newSocketUrl,
                        newMqttUrl)
                    AmityLog.tag(TAG).e("deleting user database")
                    AmityCoreClient.logout()
                    httpUrlDao.insert(EkoHttpUrl.create(newHttpUrl))
                    socketUrlDao.insert(EkoSocketUrl.create(newSocketUrl))
                    mqttUrlDao.insert(EkoMqttUrl.create(newMqttUrl))
                    apiKeyDao.insert(EkoApiKey.create(apiKey))
                }
                AmityCoreSDKInitializer.initApiService()
            }
            .doOnComplete {
                httpUrlDao.insert(EkoHttpUrl.create(newHttpUrl))
                socketUrlDao.insert(EkoSocketUrl.create(newSocketUrl))
                mqttUrlDao.insert(EkoMqttUrl.create(newMqttUrl))
                apiKeyDao.insert(EkoApiKey.create(apiKey))
                AmityCoreSDKInitializer.initApiService()
            }
            .flatMapCompletable { Completable.complete() }
            .doOnError {
                AmityLog.tag("error").e(it)
            }
            .subscribe(result)

        setupSessionComponents()
        AmityChannelReader.init()

        return result.hide()
    }

    private fun setupSessionComponents() {
        //setup session components
        //this is necessary to check if the sessionComponent instances
        //are initiated or not, so that we won't re-create mqtt/socket
        //instances again.
        if(sessionLifeCycleEventBus == null) {
            sessionLifeCycleEventBus = SessionLifeCycleEventBus()
        }
        if(appEventBus == null) {
            appEventBus = AppEventBus()
        }
        if(sessionStateEventBus == null) {
            sessionStateEventBus = SessionStateEventBus()
        }
        if(sessionStateManager == null) {
            sessionStateManager = SessionStateManager(
                appEventBus = appEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!
            )
        }

        if (mqttClient == null || socketClient == null || renewalManager == null) {
            mqttClient = AmityMqttClient(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            socketClient = EkoSocket(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            renewalManager = TokenRenewalSessionComponent(
                appEventBus = appEventBus!!,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            EkoApi(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            DatabaseSessionComponent(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            UserSettingSessionComponent(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            TokenWatcherSessionComponent(
                appEventBus = appEventBus!!,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
        }
    }

}