package com.amity.socialcloud.sdk.infra.db

import android.content.Context
import net.sqlcipher.database.SQLiteDatabase
import java.io.File
import java.io.FileNotFoundException
import java.io.IOException


object DBEncryptionUtils {
    /**
     * The detected state of the database, based on whether we can open it
     * without a passphrase.
     */
    enum class State {
        DOES_NOT_EXIST, UNENCRYPTED, ENCRYPTED
    }

    fun getDatabaseState(context: Context, dbPath: File): State {
        SQLiteDatabase.loadLibs(context)

        if (dbPath.exists()) {
            var db: SQLiteDatabase? = null

            return try {
                db = SQLiteDatabase.openDatabase(
                    dbPath.absolutePath,
                    "",
                    null,
                    SQLiteDatabase.OPEN_READONLY
                )
                db.version
                State.UNENCRYPTED
            } catch (e: Exception) {
                State.ENCRYPTED
            } finally {
                db?.close()
            }
        }

        return State.DOES_NOT_EXIST
    }

    fun isPassPhraseCorrect(context: Context, dbPath: File, passphrase: ByteArray?): Boolean {
        SQLiteDatabase.loadLibs(context)

        if (dbPath.exists()) {
            var db: SQLiteDatabase? = null
            return try {
                db = SQLiteDatabase.openDatabase(
                    dbPath.absolutePath,
                    passphrase,
                    null,
                    SQLiteDatabase.OPEN_READONLY,
                    null,
                    null
                )
                db.version
                true
            } catch (e: Exception) {
                false
            } finally {
                db?.close()
            }
        }

        return false
    }

    fun encrypt(context: Context,
                dbFile: File,
                passphrase: ByteArray
    ) {
        val dbTemp = context.getDatabasePath("_temp.db")
        dbTemp.delete()

        encryptTo(context, dbFile, dbTemp, passphrase)

        val dbBackup = context.getDatabasePath("_backup.db")

        if (dbFile.renameTo(dbBackup)) {
            if (dbTemp.renameTo(dbFile)) {
                dbBackup.delete()
            } else {
                dbBackup.renameTo(dbFile)
                throw IOException("Could not rename $dbTemp to $dbFile")
            }
        } else {
            dbTemp.delete()
            throw IOException("Could not rename $dbFile to $dbBackup")
        }
    }

    private fun encryptTo(
        context: Context,
        originalFile: File,
        targetFile: File,
        passphrase: ByteArray?
    ) {
        SQLiteDatabase.loadLibs(context)

        if (originalFile.exists()) {
            val originalDb = SQLiteDatabase.openDatabase(
                originalFile.absolutePath,
                "",
                null,
                SQLiteDatabase.OPEN_READWRITE
            )
            val version = originalDb.version

            originalDb.close()

            val db = SQLiteDatabase.openOrCreateDatabase(
                targetFile.absolutePath,
                passphrase,
                null
            )

            //language=text
            val st = db.compileStatement("ATTACH DATABASE ? AS plaintext KEY ''")

            st.bindString(1, originalFile.absolutePath)
            st.execute()
            db.rawExecSQL("SELECT sqlcipher_export('main', 'plaintext')")
            db.rawExecSQL("DETACH DATABASE plaintext")
            db.version = version
            st.close()
            db.close()
        } else {
            throw FileNotFoundException(originalFile.absolutePath + " not found")
        }
    }

    @Throws(IOException::class)
    private fun decryptTo(
        context: Context,
        originalFile: File,
        targetFile: File,
        passphrase: ByteArray?
    ) {
        SQLiteDatabase.loadLibs(context)

        if (originalFile.exists()) {
            val originalDb = SQLiteDatabase.openDatabase(
                originalFile.absolutePath,
                passphrase,
                null,
                SQLiteDatabase.OPEN_READWRITE,
                null,
                null
            )

            SQLiteDatabase.openOrCreateDatabase(
                targetFile.absolutePath,
                "",
                null
            ).close() // create an empty database

            //language=text
            val st =
                originalDb.compileStatement("ATTACH DATABASE ? AS plaintext KEY ''")

            st.bindString(1, targetFile.absolutePath)
            st.execute()
            originalDb.rawExecSQL("SELECT sqlcipher_export('plaintext')")
            originalDb.rawExecSQL("DETACH DATABASE plaintext")

            val version = originalDb.version

            st.close()
            originalDb.close()

            val db = SQLiteDatabase.openOrCreateDatabase(
                targetFile.absolutePath,
                "",
                null
            )

            db.version = version
            db.close()
        } else {
            throw FileNotFoundException(originalFile.absolutePath + " not found")
        }
    }
}