package com.amity.socialcloud.sdk.infra.mqtt

import android.util.Log
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.events.AmityTopic
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.infra.mqtt.listener.MqttEventListeners
import com.amity.socialcloud.sdk.infra.mqtt.payload.MqttPayload
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.ekosdk.internal.api.EkoEndpoint
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.hivemq.client.mqtt.MqttClient
import com.hivemq.client.mqtt.MqttGlobalPublishFilter
import com.hivemq.client.mqtt.datatypes.MqttQos
import com.hivemq.client.mqtt.mqtt3.Mqtt3RxClient
import com.hivemq.client.mqtt.mqtt3.exceptions.Mqtt3ConnAckException
import com.hivemq.client.mqtt.mqtt3.message.connect.Mqtt3Connect.DEFAULT_KEEP_ALIVE
import com.hivemq.client.mqtt.mqtt3.message.connect.connack.Mqtt3ConnAckReturnCode
import com.hivemq.client.mqtt.mqtt3.message.subscribe.Mqtt3Subscribe
import com.hivemq.client.mqtt.mqtt3.message.unsubscribe.Mqtt3Unsubscribe
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import timber.log.Timber

internal class AmityMqttClient(
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) :
    SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {


    data class AuthenticatedMqttClient(
        val clientId: String,
        val account: EkoAccount,
        val mqttClient: Mqtt3RxClient
    )

    override fun onSessionStateChange(sessionState: SessionState) {
        Log.e("SSM3", "mqtt session change: $sessionState")
    }

    override fun establish(account: EkoAccount) {
        Log.e("SSM3", "mqtt session establish: ${this.hashCode()}")
        //always disconnect mqtt first, prevent the case that
        //user login with the same user
        activeClient?.let { obsoleteClient(it) }

        io.reactivex.rxjava3.core.Completable.fromAction {
            connect(account)
        }.subscribeOn(io.reactivex.rxjava3.schedulers.Schedulers.io())
            .subscribe()

    }

    override fun destroy() {
        activeClient?.let { obsoleteClient(it) }
    }

    override fun handleTokenExpire() {
        activeClient?.let { obsoleteClient(it) }
    }

    fun disconnect(): Completable {
        return if (activeClient != null) {
            Completable.fromAction { obsoleteClient(activeClient!!) }
        } else {
            Completable.complete()
        }
    }

    private fun connect(account: EkoAccount) {
        val clientId = generateClientId(account)
        val userDao = UserDatabase.get().userDao()
        val user = userDao.getByIdNow(account.userId)
        val username = user?.mid ?: ""
        val password = account.accessToken.toByteArray()
        val mqttClient = try {
            initMqttClient(clientId, username, password)
        } catch(e: Exception) {
            // To avoid crash from Android 11 issue from this topic
            // https://issuetracker.google.com/issues/204913332
            // Which is inside HiveMQ. So, In order to handle the issue from SDK side.
            // When fail to initialize, the MQTT client will be null
            null
        }

        Log.e(TAG, "Connecting client: " + clientId + " userId: " + account.userId)

        mqttClient?.let {
            mqttClient.connectWith()
                .cleanSession(false)
                .keepAlive(DEFAULT_KEEP_ALIVE)
                .applyConnect()
                .ignoreElement()
                .doOnComplete {
                    val authClient = AuthenticatedMqttClient(clientId, account, mqttClient)
                    activeClient = authClient
                    addClientListeners(authClient)
                    autoSubscribe()
                }
                .doOnError {
                    Log.e(TAG, "Connection exception: " + it.message)
                    //Timber.tag(TAG).e("Connection exception: " + it.message)
                }
                .subscribe()
        }
    }

    private fun generateClientId(newAccount: EkoAccount): String {
        return newAccount.deviceId
    }

    private fun initMqttClient(clientId: String, username: String, password: ByteArray): Mqtt3RxClient {
        return MqttClient.builder()
            .useMqttVersion3()
            .identifier(clientId)
            .serverHost(EkoEndpoint.getMqttUrl())
            .serverPort(443)
            .sslWithDefaultConfig()
            .simpleAuth()
            .username(username)
            .password(password)
            .applySimpleAuth()
            .automaticReconnectWithDefaultConfig()
            .addConnectedListener {
                //Timber.tag(TAG).d( "clientId: " + clientId + " state: connected")
                Log.e(TAG, "mqtt connected")
            }
            .addDisconnectedListener {
                val exception = it.cause
                val isNotActive = activeClient?.clientId != clientId
                val hasValidDisconnectReason = exception is Mqtt3ConnAckException
                        &&
                        (exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.IDENTIFIER_REJECTED
                                || exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.UNSUPPORTED_PROTOCOL_VERSION
                                || exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.BAD_USER_NAME_OR_PASSWORD
                                || exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.NOT_AUTHORIZED)

                val shouldNotReconnect = isNotActive || hasValidDisconnectReason
                if (shouldNotReconnect) {
                    it.reconnector.reconnect(false)
                }
                Log.e(
                    TAG,
                    "mqtt disconnected || isNotActive: $isNotActive || hasValidDisconnectReason: $hasValidDisconnectReason "
                )
                Log.e(TAG, "mqtt disconnected || cause: $exception")

                //Timber.tag(TAG).d( "clientId: " + clientId + " state: disconnected: " + exception.message)
            }
            .buildRx()
    }

    //obsolete will disconnect and terminate the current mqtt client
    private fun obsoleteClient(authClient: AuthenticatedMqttClient) {
        //Timber.tag(TAG).d( "Disconnecting client: " + authClient.clientId + " userId: " + authClient.account.userId)
        subscriptions.clear()
        Log.e(
            TAG,
            "Disconnecting client: \" + ${authClient.clientId} + \" userId : \" + ${authClient.account.userId}"
        )
        val mqttClient = authClient.mqttClient
        mqttClient.disconnect()
            .subscribeOn(Schedulers.io())
            .doOnError {
                Log.e(TAG, "Disconnect error: " + it.message)
                //Timber.tag(TAG).d( "Disconnect error: " + it.message)
            }
            .subscribe()

        activeClient = null
    }

    private fun createEventSubscription(authClient: AuthenticatedMqttClient): Disposable {
        val mqttClient = authClient.mqttClient
        return mqttClient
            .publishes(MqttGlobalPublishFilter.ALL)
            .subscribeOn(Schedulers.io())
            .doOnNext {
                val isNotActiveClient = authClient.clientId != activeClient?.clientId
                if (isNotActiveClient) {
                    obsoleteClient(authClient)
                    return@doOnNext
                }
                try {
                    val payload = String(it.payloadAsBytes)
                    Timber.tag(TAG).d("received event: " + payload)
                    val event = EkoGson.get().fromJson(payload, MqttPayload::class.java)
                    val listener = MqttEventListeners.getMap()[event.eventType]
                    if (listener != null) {
                        event.data?.let { data ->
                            listener.onEvent(data)
                        }
                    } else {
                        //Timber.tag(TAG).e( "Listener not found for event: " + event.eventType)
                    }
                } catch (exception: Exception) {
                    //Timber.tag(TAG).e( "Payload processing error: " + exception.message)
                }

            }
            .doOnError {
                //Timber.tag(TAG).e( "Unexpected Mqtt error")
            }
            .subscribe()
    }

    private fun addClientListeners(authClient: AuthenticatedMqttClient) {
        val eventSubscription = createEventSubscription(authClient)
        subscriptions.add(eventSubscription)
    }

    private fun autoSubscribe() {
        val networkSubscription = subscribe(AmityTopic.NETWORK())
            .subscribeOn(Schedulers.io())
            .doOnError {
                Timber.tag(TAG).e("Failed to subscribe network events")
            }
            .subscribe()
        subscriptions.add(networkSubscription)
    }

    companion object {
        private val subscriptions = CompositeDisposable()
        private var activeClient: AuthenticatedMqttClient? = null
        val TAG = "AmityMqtt"

        fun subscribe(mqttTopic: AmityTopic): Completable {
            return mqttTopic.generateTopic()
                .flatMapCompletable { topic ->
                    getCurrentClient().flatMapCompletable { client ->
                        val subscribeMessage = Mqtt3Subscribe.builder()
                            .topicFilter(topic)
                            .qos(MqttQos.AT_LEAST_ONCE)
                            .build()
                        client.subscribe(subscribeMessage)
                            .doOnSuccess {
                                Timber.tag(TAG).d("Subscribed to $topic")
                            }
                            .onErrorResumeNext {
                                Timber.tag(TAG).e("Failed to subscribe $topic")
                                val exception = AmityException.create(
                                    "Failed to subscribe",
                                    null,
                                    AmityError.UNKNOWN
                                )
                                Single.error(exception)
                            }
                            .ignoreElement()
                    }
                }

        }

        fun unsubscribe(mqttTopic: AmityTopic): Completable {
            return mqttTopic.generateTopic()
                .flatMapCompletable { topic ->
                    getCurrentClient()
                        .flatMapCompletable {
                            val unsubscribeMessage = Mqtt3Unsubscribe.builder()
                                .topicFilter(topic)
                                .build()
                            it.unsubscribe(unsubscribeMessage)
                        }
                }
        }

        private fun getCurrentClient(): Single<Mqtt3RxClient> {
            val client = activeClient?.mqttClient
            return if (client == null) {
                val exception =
                    AmityException.create(
                        "Failed to subscribe",
                        null,
                        AmityError.UNKNOWN
                    )
                Single.error(exception)
            } else {
                Single.just(client)
            }
        }
    }
}