package com.amity.socialcloud.sdk.social.data.category

import androidx.paging.*
import androidx.arch.core.util.Function
import androidx.paging.PagedList
import androidx.paging.rxjava3.flowable
import com.amity.socialcloud.sdk.social.community.AmityCommunityCategory
import com.amity.socialcloud.sdk.social.community.AmityCommunityCategorySortOption
import com.amity.socialcloud.sdk.social.domain.category.CategoryComposerUseCase
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoCommunityCategoryBoundaryCallback
import com.ekoapp.ekosdk.internal.repository.community.CommunityCategoryRxRemoteMediator
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.subjects.PublishSubject
import java.util.concurrent.Executors

@OptIn(ExperimentalPagingApi::class)
internal class CategoryRepository : EkoObjectRepository() {

    fun getCategories(categoryIds: List<String>): List<AmityCommunityCategory> {
        return CategoryLocalDataStore().getCategories(categoryIds)
            .map {
                CategoryModelMapper().map(it)
            }
    }

    fun getAllCategories(sortOption: AmityCommunityCategorySortOption, isDeleted: Boolean?): Flowable<PagedList<AmityCommunityCategory>> {
        val communityCategory = UserDatabase.get().communityCategoryDao()
        val localFactory = communityCategory.getAll(sortOption, isDeleted)
        val factory = localFactory.map(Function { CategoryModelMapper().map(it) })
            .map(Function {
                // Bad hack. To be removed ASAP
                CategoryComposerUseCase().execute(it)
            })

        val delaySubject = PublishSubject.create<Boolean>()
        val boundaryCallback = EkoCommunityCategoryBoundaryCallback(
            sortOption = sortOption,
            isDeleted = isDeleted,
            pageSize = getDefaultPageSize(),
            delaySubject = delaySubject
        )

        return createRxCollectionWithBoundaryCallback(factory.map(boundaryCallback), boundaryCallback).toV3()
    }

    fun getAllCategoriesPagingData(sortOption: AmityCommunityCategorySortOption,
                                   isDeleted: Boolean?): Flowable<PagingData<AmityCommunityCategory>> {

        val pager = Pager(
                config = PagingConfig(
                        pageSize = DEFAULT_PAGE_SIZE,
                        enablePlaceholders = false
                ),
                initialKey = null,
                remoteMediator = CommunityCategoryRxRemoteMediator(
                        sortOption = sortOption,
                        isDeleted = isDeleted,
                        communityCategoryQueryTokenDao = UserDatabase.get().communityCategoryQueryTokenDao()
                )
        ) {
            val communityCategoryDao = UserDatabase.get().communityCategoryDao()
            communityCategoryDao.getPagingSource(sortOption, isDeleted ?: false)
        }

        return pager.flowable
                .map { pagingData ->
                    pagingData.map(Executors.newSingleThreadExecutor()) {
                        CategoryModelMapper().map(it)
                    }
                }
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

}