package com.amity.socialcloud.sdk.social.data.comment

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.social.comment.AmityComment
import com.amity.socialcloud.sdk.social.comment.AmityCommentSortOption
import com.amity.socialcloud.sdk.social.data.post.PostLocalDataStore
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.dto.EkoMentioneesDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommentQueryRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CommentUpdateRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CreateCommentRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoCommentBoundaryCallback
import com.ekoapp.ekosdk.internal.data.model.EkoCommentFlagEntity
import com.ekoapp.ekosdk.internal.entity.CommentEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.comment.CommentLoadResult
import com.ekoapp.ekosdk.internal.repository.comment.CommentMediator
import com.ekoapp.ekosdk.internal.repository.comment.helper.EkoCommentFactory
import com.google.common.base.Objects
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.subjects.PublishSubject
import org.amity.types.ObjectId
import org.joda.time.DateTime

@OptIn(ExperimentalPagingApi::class)
internal class CommentRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun getComment(commentId: String): AmityComment? {
        return CommentLocalDataStore().getComment(commentId)?.let {
            CommentModelMapper().map(it)
        }
    }

    fun getLatestComments(
        referenceType: String,
        referenceId: String
    ): List<AmityComment> {
        return CommentLocalDataStore().getLatestComments(referenceType, referenceId)
    }

    fun getLatestReplies(
        referenceType: String,
        referenceId: String,
        parentId: String
    ): List<AmityComment> {
        return CommentLocalDataStore().getLatestReplies(
            referenceType = referenceType,
            referenceId = referenceId,
            parentId = parentId,
        )
    }

    fun observeComment(commentId: String): Flowable<AmityComment> {
        return CommentLocalDataStore().observeComment(commentId)
            .map {
                CommentModelMapper().map(it)
            }
    }

    fun createComment(
        referenceType: String,
        referenceId: String,
        parentId: String?,
        commentId: String?,
        data: JsonObject?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): Single<AmityComment> {

        val comment = createTempComment(
            referenceType = referenceType,
            referenceId = referenceId,
            parentId = parentId,
            commentId = commentId,
            data = data,
            metadata = metadata,
            mentionees = mentionees
        )
        val commentFlag = createTempFlag(commentId = comment.commentId)

        val request = CreateCommentRequest(
            commentId = comment.commentId,
            referenceType = referenceType,
            referenceId = referenceId,
            parentId = parentId,
            data = data,
            metadata = metadata,
            mentionees = mentionees
        )

        return Single.fromCallable {
            insertOrUpdate(comment, commentFlag, AmityComment.State.SYNCING.stateName)
            comment
        }.flatMap {
            CommentRemoteDataStore().createComment(request)
        }.onErrorResumeNext {
            updateSyncState(comment, AmityComment.State.FAILED.stateName)
            Single.error(it)
        }.flatMap {
            CommentQueryPersister().persist(it)
                .andThen(Single.just(it.getComments().get(0).getCommentId()))
        }.map {
            val comment = getComment(it)
            comment?.getReference()?.let { reference ->
                if (reference is AmityComment.Reference.POST) {
                    val postLocalDataStore = PostLocalDataStore()
                    postLocalDataStore.incrementCommentCount(reference.getPostId())
                    postLocalDataStore.notifyPost(reference.getPostId())
                } else if (reference is AmityComment.Reference.CONTENT) {
                    val postLocalDataStore = PostLocalDataStore()
                    postLocalDataStore.incrementCommentCount(reference.getContentId())
                    postLocalDataStore.notifyPost(reference.getContentId())
                }
            }
            comment?.getParentId()?.let {
                val commentLocalDataStore = CommentLocalDataStore()
                commentLocalDataStore.notifyParentComment(it)
            }
            comment
        }
    }

    fun editComment(
        commentId: String,
        data: JsonObject?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): Single<AmityComment> {
        val request = CommentUpdateRequest(commentId, data, metadata, mentionees)
        return CommentRemoteDataStore().updateComment(commentId, request)
            .flatMap {
                CommentQueryPersister().persist(it)
                    .andThen(Single.just(it.getComments().get(0).getCommentId()))
            }.map {
                val comment = getComment(it)
                comment?.getReference()?.let {
                    if (it is AmityComment.Reference.POST) {
                        val postLocalDataStore = PostLocalDataStore()
                        postLocalDataStore.notifyPost(it.getPostId())
                    }
                }
                comment?.getParentId()?.let {
                    val commentLocalDataStore = CommentLocalDataStore()
                    commentLocalDataStore.notifyParentComment(it)
                }
                comment
            }
    }

    fun deleteComment(commentId: String, hardDelete: Boolean): Completable {
        val userDatabase = UserDatabase.get()
        val commentDao = userDatabase.commentDao()
        return commentDao.getById(commentId)
            .firstOrError()
            .subscribeOn(Schedulers.io())
            .flatMapCompletable { localComment ->
                if (localComment.syncState == AmityComment.State.FAILED.stateName) {
                    commentDao.delete(localComment)
                    Completable.complete()
                } else {
                    CommentRemoteDataStore().deleteComment(commentId, hardDelete)
                        .flatMapCompletable {
                            if (it.isSuccess()) {
                                val commentLocalDataStore = CommentLocalDataStore()
                                val commentEntity = commentDao.getByIdNow(commentId)
                                if (commentEntity != null) {
                                    if (Objects.equal(commentEntity.referenceType, "post")) {
                                        PostLocalDataStore().decrementCommentCount(commentEntity.referenceId)
                                    } else if (Objects.equal(commentEntity.referenceType, "content")) {
                                        PostLocalDataStore().decrementCommentCount(commentEntity.referenceId)
                                    }
                                    if (hardDelete) {
                                        commentLocalDataStore.hardDelete(commentId)
                                    } else {
                                        commentLocalDataStore.softDelete(commentId)
                                    }

                                } else {
                                    Completable.complete()
                                }
                            } else {
                                Completable.complete()
                            }
                        }
                }
            }
    }

    fun getCommentCollection(
        referenceId: String,
        referenceType: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption
    ): io.reactivex.Flowable<PagedList<AmityComment>> {
        val factory = EkoCommentFactory().getFactory(
            referenceId,
            referenceType,
            isFilterByParentId,
            parentId,
            isDeleted,
            sortOption
        )
        val delaySubject = PublishSubject.create<Boolean>()
        val boundaryCallback = EkoCommentBoundaryCallback(
            referenceId = referenceId,
            referenceType = referenceType,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            pageSize = getDefaultPageSize(),
            sortBy = sortOption,
            delaySubject = delaySubject
        )

        return createRxCollectionWithBoundaryCallback(
            factory.map(boundaryCallback),
            boundaryCallback
        )
    }

    fun getCommentPagingData(
        referenceId: String,
        referenceType: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption
    ): Flowable<PagingData<AmityComment>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = CommentMediator(
                referenceId = referenceId,
                referenceType = referenceType,
                parentId = parentId,
                isDeleted = isDeleted,
                sortOption = sortOption,
                isFilterByParentId = isFilterByParentId
            ),
            pagingSourceFactory = {
                CommentLocalDataStore().getCommentPagingSource(
                    referenceId = referenceId,
                    referenceType = referenceType,
                    isFilterByParentId = isFilterByParentId,
                    parentId = parentId,
                    isDeleted = isDeleted,
                    sortOption = sortOption
                )
            },
            modelMapper = CommentModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun getLatestComment(
        referenceId: String,
        referenceType: String,
        isFilterByParentId: Boolean
    ): Single<AmityComment> {
        val options = CommentQueryRequest.CommentQueryRequestOptions(limit = 1, type = "pagination")
        val commentQueryRequest = CommentQueryRequest(
            referenceId = referenceId,
            referenceType = referenceType,
            filterByParentId = isFilterByParentId,
            isDeleted = false,
            sortBy = AmityCommentSortOption.LAST_CREATED.apiKey,
            options = options
        )

        return CommentRemoteDataStore().queryComments(commentQueryRequest)
            .flatMap {
                CommentQueryPersister().persist(it).andThen(Single.just(it))
            }
            .flatMap {
                if (it.comments.isEmpty()) {
                    val exception =
                        AmityException.create("item not found", null, AmityError.ITEM_NOT_FOUND)
                    Single.error(exception)
                } else {
                    val commentId = it.comments.first().commentId
                    val comment: CommentEntity? = CommentLocalDataStore().getComment(commentId)
                    if (comment == null) {
                        val exception =
                            AmityException.create("item not found", null, AmityError.ITEM_NOT_FOUND)
                        Single.error(exception)
                    } else {
                        Single.just(CommentModelMapper().map(comment))
                    }
                }
            }
    }

    fun observeCommentAfter(
        referenceType: String,
        referenceId: String,
        parentId: String?,
        isFilterByParentId: Boolean,
        sortOption: AmityCommentSortOption,
        offsetCommentId: String?
    ): Flowable<List<AmityComment>> {

        return UserDatabase.get().commentDao().observeCommentAfter(
            referenceType,
            referenceId,
            parentId,
            isFilterByParentId,
            sortOption,
            offsetCommentId
        ).map {
            it.map {
                CommentModelMapper().map(it)
            }
        }
    }

    fun loadComments(
        referenceId: String,
        referenceType: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        token: String
    ): Single<CommentLoadResult> {
        val options = CommentQueryRequest.CommentQueryRequestOptions().apply {
            this.type = "pagination"
            this.token = token
        }
        val commentQueryRequest = CommentQueryRequest(
            referenceId = referenceId,
            referenceType = referenceType,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            options = options
        )
        return CommentRemoteDataStore().queryComments(commentQueryRequest)
            .flatMap {
                CommentQueryPersister().persist(it).andThen(Single.just(it))
            }
            .map { commentAndUserListDto ->
                val ids = commentAndUserListDto.comments.map { it.commentId }
                val tokenNext = commentAndUserListDto.token?.next ?: ""
                CommentLoadResult(tokenNext, ids)
            }
    }

    fun loadFirstPageComments(
        referenceType: String,
        referenceId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption,
        limit: Int
    ): Single<CommentLoadResult> {

        val options = CommentQueryRequest.CommentQueryRequestOptions().apply {
            this.skip = 0
            this.limit = limit
            this.type = "pagination"
        }
        val commentQueryRequest = CommentQueryRequest(
            referenceId = referenceId,
            referenceType = referenceType,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            sortBy = sortOption.apiKey,
            options = options
        )

        return CommentRemoteDataStore().queryComments(commentQueryRequest)
            .flatMap {
                CommentQueryPersister().persist(it).andThen(Single.just(it))
            }
            .map { commentAndUserListDto ->
                val ids = commentAndUserListDto.comments.map { it.commentId }
                val token = commentAndUserListDto.token?.next ?: ""
                CommentLoadResult(token, ids)
            }
    }

    fun markDeletedAfterCommentId(commentId: String) {
        CommentLocalDataStore().markDeletedAfterCommentId(commentId)
    }

    fun markDeletedBeforeCommentId(commentId: String) {
        CommentLocalDataStore().markDeletedBeforeCommentId(commentId)
    }

    fun getCommentCollection(
        ids: List<String>,
        sortOption: AmityCommentSortOption
    ): Flowable<List<AmityComment>> {
        return UserDatabase.get().commentDao().getAllByIds(ids, sortOption)
            .map { list ->
                val commentList = mutableListOf<AmityComment>()
                list.forEach { commentEntity ->
                    commentList.add(CommentModelMapper().map(commentEntity))
                }
                commentList
            }
    }

    fun flagComment(commentId: String): Completable {
        return CommentRemoteDataStore().flagComment(commentId)
            .flatMapCompletable {
                CommentQueryPersister().persist(it)
            }
    }

    fun unflagComment(commentId: String): Completable {
        return CommentRemoteDataStore().unflagComment(commentId)
            .flatMapCompletable {
                CommentQueryPersister().persist(it)
            }
    }

    fun isFlaggedByMe(commentId: String): Single<JsonObject> {
        return CommentRemoteDataStore().isFlaggedByMe(commentId)
    }

    fun getLatestComment(
        referenceId: String,
        referenceType: String,
        parentId: String?,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityComment> {
        return CommentLocalDataStore()
            .getLatestComment(
                referenceId = referenceId,
                referenceType = referenceType,
                parentId = parentId,
                isDeleted = isDeleted,
                dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
                nonce = nonce
            ).map {
                CommentModelMapper().map(it)
            }
    }

    private fun createTempComment(
        referenceType: String,
        referenceId: String,
        parentId: String?,
        commentId: String?,
        data: JsonObject?,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): CommentEntity {
        val id = if (commentId.isNullOrBlank()) ObjectId.get().toHexString() else commentId
        val now = DateTime()
        val comment = CommentEntity(
            commentId = id,
            referenceType = referenceType,
            referenceId = referenceId,
            parentId = parentId,
            data = data,
            userId = AmityCoreClient.getUserId(),
            syncState = AmityComment.State.CREATED.stateName,
            editedAt = now,
            metadata = metadata,
            mentionees = getMentioneesDto(mentionees) ?: listOf()
        )
        comment.createdAt = now
        comment.updatedAt = now
        return comment
    }

    private fun getMentioneesDto(mentionees: List<AmityMentioneeTarget>?): List<EkoMentioneesDto>? {
        return mentionees?.map { EkoMentioneesDto(it.type, it.userIds) }
    }

    private fun createTempFlag(commentId: String): EkoCommentFlagEntity {
        val commentFlag = EkoCommentFlagEntity()
        commentFlag.commentId = commentId
        return commentFlag
    }

    private fun insertOrUpdate(
        comment: CommentEntity,
        commentFlag: EkoCommentFlagEntity,
        syncState: String
    ) {
        val commentDao = UserDatabase.get().commentDao()
        val commentFlagDao = UserDatabase.get().commentFlagDao()
        commentDao.insert(comment)
        commentFlagDao.insert(commentFlag)
        updateSyncState(comment, syncState)
    }

    private fun updateSyncState(comment: CommentEntity, syncState: String) {
        val commentDao = UserDatabase.get().commentDao()
        comment.syncState = syncState
        commentDao.update(comment)
        comment.syncState = AmityComment.State.FAILED.stateName
        commentDao.update(comment)
    }

}