package com.amity.socialcloud.sdk.social.data.community

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.social.community.AmityCommunitySortOption
import com.amity.socialcloud.sdk.social.feed.AmityFeedType
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.entity.FeedEntity
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class CommunityLocalDataStore {

    // To rename to "getCommunity"
    fun getCommunityById(communityId: String): CommunityEntity? {
        var community: CommunityEntity? = null
        Completable.fromCallable {
            val communityDao = UserDatabase.get().communityDao()
            val entity = communityDao.getByIdNow(communityId)
            if (entity != null) {
                community = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return community
    }

    fun getCommunities(ids: List<String>): Flowable<List<CommunityEntity>> {
        return UserDatabase.get().communityDao().observeByCommunityIds(ids)
            .map { entities ->
                val orderById = ids.withIndex().associate { it.value to it.index }
                val sortedCommunities = entities.sortedBy { orderById[it.communityId] }
                sortedCommunities
            }
    }

    fun observeCommunity(communityId: String): Flowable<CommunityEntity> {
        return UserDatabase.get().communityDao().getByCommunityId(communityId)
    }

    fun saveCommunities(communities: List<CommunityEntity>): Completable {
        return Completable.fromAction { UserDatabase.get().communityDao().save(communities) }
    }

    fun deleteCommunity(communityId: String): Completable {
        return UserDatabase.get().communityDao().deleteByCommunityId(communityId)
    }

    fun getPostCount(targetId: String, feedType: AmityFeedType): Flowable<Int> {
        val feedDao = UserDatabase.get().feedDao()
        return feedDao.getCommunityFeed(targetId, feedType.apiKey).map(FeedEntity::postCount)
    }

    fun hasInLocal(communityId: String): Boolean {
        val communityDao = UserDatabase.get().communityDao()
        return communityDao.getByIdNow(communityId) != null
    }
    
    fun getCommunityPagingSource(
            keyword: String,
            categoryId: String,
            filter: AmityCommunityFilter,
            sortBy: AmityCommunitySortOption,
            isDeleted: Boolean?
    ): PagingSource<Int, CommunityEntity> {
        return UserDatabase.get().communityPagingDao().queryCommunities(
                keyword = keyword,
                categoryId = categoryId,
                filter = filter,
                sortBy = sortBy,
                isDeleted = isDeleted
        )
    }

}