package com.amity.socialcloud.sdk.social.data.community

import androidx.paging.*
import androidx.paging.rxjava3.flowable
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunity
import com.amity.socialcloud.sdk.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.social.community.AmityCommunitySortOption
import com.amity.socialcloud.sdk.social.domain.community.CommunityComposerUseCase
import com.amity.socialcloud.sdk.social.feed.AmityFeedType
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.AmityObjectRepository
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoCommunityBoundaryCallback
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.repository.community.CommunityPageKeyedRxRemoteMediator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.subjects.PublishSubject
import java.util.concurrent.Executors

@OptIn(ExperimentalPagingApi::class)
internal class CommunityRepository : AmityObjectRepository<CommunityEntity, AmityCommunity>() {


    override fun fetchAndSave(objectId: String): Completable {
        return CommunityRemoteDataStore().getCommunity(objectId)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): CommunityEntity? {
        return CommunityLocalDataStore().getCommunityById(objectId)
    }

    override fun mapper(): ModelMapper<CommunityEntity, AmityCommunity> {
        return CommunityModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<CommunityEntity> {
        return CommunityLocalDataStore().observeCommunity(objectId)
    }

    fun getCommunityCollection(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortBy: AmityCommunitySortOption,
        isDeleted: Boolean?
    ): Flowable<PagedList<AmityCommunity>> {

        val factory = getFactory(
            keyword = keyword,
            categoryId = categoryId,
            filter = filter,
            sortBy = sortBy,
            isDeleted = isDeleted
        )
            .map { CommunityModelMapper().map(it) }
            .map { CommunityComposerUseCase().execute(it) }

        val delaySubject = PublishSubject.create<Boolean>()
        val boundaryCallback = EkoCommunityBoundaryCallback(
            keyword = keyword,
            categoryId = categoryId,
            filter = filter,
            sortBy = sortBy,
            isDeleted = isDeleted,
            pageSize = getDefaultPageSize(),
            delaySubject = delaySubject
        )

        return createRxCollectionWithBoundaryCallback(
            factory.map(boundaryCallback),
            boundaryCallback
        ).toV3()
    }
    
    fun getCommunityPagingData(
            keyword: String,
            categoryId: String,
            filter: AmityCommunityFilter,
            sortBy: AmityCommunitySortOption,
            isDeleted: Boolean?
    ): Flowable<PagingData<AmityCommunity>> {
    
        val pager = Pager(
                config = PagingConfig(pageSize = getDefaultPageSize(), enablePlaceholders = false),
                initialKey = null,
                remoteMediator = CommunityPageKeyedRxRemoteMediator(
                        keyword = keyword,
                        categoryId = categoryId,
                        filter = filter,
                        sortBy = sortBy,
                        isDeleted = isDeleted,
                        tokenDao = UserDatabase.get().queryTokenDao()
                )
        ) {
            CommunityLocalDataStore().getCommunityPagingSource(keyword,categoryId, filter, sortBy, isDeleted)
        }
        return pager.flowable
                .map { pagingData ->
                    pagingData.map(Executors.newSingleThreadExecutor()) { community ->
                        CommunityModelMapper().map(community)
                    }.map(Executors.newSingleThreadExecutor()) { community ->
                        CommunityComposerUseCase().execute(community)
                    }
                }
    }

    fun getRecommendedCommunities(): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getRecommendedCommunities(QueryOptionsRequestParams(limit = DEFAULT_PAGE_SIZE))
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(Single.just(it).map { it.communities.map { it.communityId!! } })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

    fun getTrendingCommunities(): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getTrendingCommunities(QueryOptionsRequestParams(limit = DEFAULT_PAGE_SIZE))
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(Single.just(it).map { it.communities.map { it.communityId!! } })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

    private fun getFactory(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortBy: AmityCommunitySortOption,
        isDeleted: Boolean?
    ): DataSource.Factory<Int, CommunityEntity> {
        val communityDao = UserDatabase.get().communityDao()
        when (filter) {
            AmityCommunityFilter.MEMBER -> {
                return if (categoryId.isEmpty()) {
                    communityDao.getAllByKeywordForMember(keyword, sortBy, isDeleted)
                } else {
                    communityDao.getAllByCategoryIdForMember(keyword, categoryId, sortBy, isDeleted)
                }
            }
            AmityCommunityFilter.NOT_MEMBER -> {
                return if (categoryId.isEmpty()) {
                    communityDao.getAllByKeywordForNonMember(keyword, sortBy, isDeleted)
                } else {
                    communityDao.getAllByCategoryIdForNonMember(
                        keyword,
                        categoryId,
                        sortBy,
                        isDeleted
                    )
                }
            }
            else -> {
                return if (categoryId.isEmpty()) {
                    communityDao.getAllByKeyword(keyword, sortBy, isDeleted)
                } else {
                    communityDao.getAllByCategoryId(keyword, categoryId, sortBy, isDeleted)
                }
            }
        }
    }

    fun createCommunity(
        displayName: String,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        userIds: List<String>?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?
    ): Single<AmityCommunity> {
        return CommunityRemoteDataStore().createCommunity(
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            userIds = userIds,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost
        ).flatMap { dto ->
            CommunityQueryPersister().persist(dto)
                .andThen(
                    dto.communities.firstOrNull()?.communityId?.let { communityId ->
                        CommunityLocalDataStore().observeCommunity(communityId).firstOrError()
                            .map {  CommunityModelMapper().map(it) }
                } ?: Single.error(
                    AmityException.create(
                        "corrupted payload",
                        null,
                        AmityError.UNKNOWN
                    )
                )
            )
        }
    }


    fun updateCommunity(
        communityId: String,
        displayName: String?,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?
    ): Single<AmityCommunity> {
        return CommunityRemoteDataStore().updateCommunity(
            communityId = communityId,
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost
        ).flatMap { dto ->
            CommunityQueryPersister().persist(dto)
                .andThen(dto.communities.firstOrNull()?.communityId?.let { communityId ->
                    CommunityLocalDataStore().observeCommunity(communityId)
                        .firstOrError()
                        .map { CommunityModelMapper().map(it) }
                } ?: Single.error(
                    AmityException.create(
                        "corrupted payload",
                        null,
                        AmityError.UNKNOWN
                    )
                ))
        }
    }

    // To rename to "getCommunity"
    fun getCommunityById(communityId: String): AmityCommunity? {
        val community = CommunityLocalDataStore().getCommunityById(communityId)
        return if (community == null) null else CommunityModelMapper().map(community)
    }



    fun joinCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().joinCommunity(communityId)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun leaveCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().leaveCommunity(communityId)
            .flatMapCompletable { CommunityQueryPersister().persist(it) }
    }

    fun deleteCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().deleteCommunity(communityId)
            .flatMapCompletable {
                when (it.isSuccess) {
                    true -> CommunityLocalDataStore().deleteCommunity(communityId)
                    else -> Completable.complete()
                }
            }
    }

    fun getPostCount(targetId: String, feedType: AmityFeedType): Flowable<Int> {
        return CommunityLocalDataStore().getPostCount(targetId, feedType)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }
}