package com.amity.socialcloud.sdk.social.data.community.membership

import androidx.paging.*
import androidx.paging.rxjava3.flowable
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.social.community.AmityCommunityMember
import com.amity.socialcloud.sdk.social.community.AmityCommunityMembershipFilter
import com.amity.socialcloud.sdk.social.community.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.domain.community.member.CommunityMemberComposerUseCase
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.community.membership.query.AmityCommunityMembership
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoCommunityMembershipBoundaryCallback
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.community.CommunityMembershipMediator
import com.ekoapp.ekosdk.internal.repository.community.SearchMembershipRxRemoteMediator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

@OptIn(ExperimentalPagingApi::class)
internal class CommunityMembershipRepository : EkoObjectRepository() {

    fun addUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().addUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun removeUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().removeUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun addRoles(communityId: String, roles: List<String>, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().addRoles(communityId, roles, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun removeRoles(communityId: String, roles: List<String>, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().removeRoles(communityId, roles, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun banUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().banUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun unbanUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().unbanUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun getMembership(communityId: String, userId: String): AmityCommunityMember? {
        val member = CommunityMembershipLocalDataStore().getMembership(communityId, userId)
        return if (member == null) null else CommunityMembershipModelMapper().map(member)
    }

    fun searchCommunityMembershipPagingData(
        communityId: String,
        roles: AmityRoles,
        communityMembership: List<AmityCommunityMembership>?,
        keyword: String
    ): Flowable<PagingData<AmityCommunityMember>> {
        val pager = Pager(
            config = PagingConfig(
                pageSize = DEFAULT_PAGE_SIZE,
                enablePlaceholders = false
            ),
            initialKey = null,
            remoteMediator = SearchMembershipRxRemoteMediator(
                communityId = communityId,
                roles = roles,
                filters = communityMembership,
                keyword = keyword,
                tokenDao = UserDatabase.get().queryTokenDao()
            )
        ) {
            val communityMembershipDao = UserDatabase.get().communityMembersPagingDao()
            communityMembershipDao.searchCommunityMembership(
                communityId,
                roles,
                communityMembership,
                keyword
            )
        }
        return pager.flowable
            .map { it.map { entity -> CommunityMembershipModelMapper().map(entity) } }
    }

    fun getCommunityMembershipCollection(
        communityId: String,
        roles: List<String>?,
        filter: AmityCommunityMembershipFilter,
        sortBy: AmityCommunityMembershipSortOption
    ): Flowable<PagedList<AmityCommunityMember>> {
        val communityMembershipDao = UserDatabase.get().communityMembershipDao()
        val factory =
            communityMembershipDao.getMembers(communityId, roles, filter.memberships, sortBy)
        val boundaryCallback = EkoCommunityMembershipBoundaryCallback(
            communityId,
            filter.memberships,
            roles,
            sortBy.apiKey,
            getDefaultPageSize()
        )
        return createRxCollectionWithBoundaryCallback(factory.map { CommunityMembershipModelMapper().map(it) }
            .map { CommunityMemberComposerUseCase().execute(it) }, boundaryCallback
        ).toV3()
    }

    fun getCommunityMembershipPagingData(
        communityId: String,
        roles: List<String>?,
        filters: AmityCommunityMembershipFilter,
        sortBy: AmityCommunityMembershipSortOption
    ): Flowable<PagingData<AmityCommunityMember>> {
        val roles = roles?.let { AmityRoles(it) } ?: AmityRoles()
        val memberships = filters.memberships.map(AmityCommunityMembership::fromApiKey)
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            queryStreamMediator = CommunityMembershipMediator(
                communityId,
                roles,
                memberships,
                sortBy
            ),
            pagingSourceFactory = {
                CommunityMembershipLocalDataStore().getMembershipPagingSource(
                    communityId,
                    roles,
                    memberships,
                    sortBy.apiKey
                )
            },
            modelMapper = CommunityMembershipModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }
}