package com.amity.socialcloud.sdk.social.data.post

import androidx.paging.DataSource
import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.social.feed.AmityCommunityFeedSortOption
import com.amity.socialcloud.sdk.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.social.feed.AmityPost
import com.amity.socialcloud.sdk.social.feed.AmityUserFeedSortOption
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class PostLocalDataStore {

    fun hasInLocal(postId: String): Boolean {
        return UserDatabase.get().postDao().getByIdNow(postId) != null
    }

    fun savePosts(posts: List<PostEntity>): Completable {
        return Completable.fromAction { UserDatabase.get().postDao().save(posts) }
    }

    fun updateFeedType(postId: String, feedType: AmityFeedType) {
        return UserDatabase.get().postDao().updateFeedType(postId, feedType.apiKey)
    }

    fun getFeedType(postId: String): AmityFeedType {
        var feedType = AmityFeedType.NONE
        Completable.fromCallable {
            val post = UserDatabase.get().postDao().getByIdNow(postId)
            if (post != null) {
                feedType = AmityFeedType.enumOf(post.feedType)
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()

        return feedType
    }

    fun observe(postId: String): Flowable<PostEntity> {
        return UserDatabase.get().postDao().observePost(postId)
            .map {
                if (it.isEmpty()) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it[0]
                }
            }
    }

    fun hardDelete(postId: String): Completable {
        return UserDatabase.get().postDao().deleteById(postId)
    }

    fun softDelete(postId: String): Completable {
        return UserDatabase.get().postDao().softDeleteById(postId)
    }

    fun getPost(postId: String): PostEntity? {
        var post: PostEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().postDao().getByIdNow(postId)
            if (entity != null) {
                post = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return post
    }

    fun getPosts(postIds: List<String>): List<PostEntity> {
        var posts = emptyList<PostEntity>()
        Completable.fromCallable {
            val postDao = UserDatabase.get().postDao()
            val entities = postDao.getByIdsNow(postIds)
            if (entities.isNotEmpty()) {
                posts = entities.let { children ->
                    val orderById = postIds.withIndex().associate { it.value to it.index }
                    val sortedPost = children.sortedBy { orderById[it.postId] }
                    sortedPost
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return posts
    }

    fun getGlobalPostsDataSourceFactory(): DataSource.Factory<Int, PostEntity> {
        return UserDatabase.get().postDao().globalPostDataSource
    }

    fun getCommunityPostsDataSourceFactory(
        communityId: String,
        sortOption: AmityCommunityFeedSortOption,
        isDeleted: Boolean?,
        feedType: AmityFeedType
    ): DataSource.Factory<Int, PostEntity> {
        return UserDatabase.get().postDao()
            .getCommunityFeedDataSource(communityId, feedType.apiKey, isDeleted, sortOption)
    }

    fun getUserPostsDataSourceFactory(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?
    ): DataSource.Factory<Int, PostEntity> {
        return UserDatabase.get().postDao().getUserFeedDataSource(userId, isDeleted, sortOption)
    }

    fun getCommunityPostsPagingSource(
        communityId: String,
        sortOption: AmityCommunityFeedSortOption,
        feedType: AmityFeedType,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf()
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().communityPostPagingDao().getCommunityPostPagingSource(
            communityId,
            feedType,
            isDeleted,
            sortOption.apiKey,
            postTypes
        )
    }

    fun getUserPostsPagingSource(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf()
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().userPostPagingDao().getUserFeedPagingSource(
            userId,
            isDeleted,
            sortOption.apiKey,
            postTypes
        )
    }

    fun getGlobalPostsPagingSource(): PagingSource<Int, PostEntity> {
        return UserDatabase.get().globalFeedPagingDao().queryGlobalFeedPost()
    }

    fun getGlobalPostsV5PagingSource(): PagingSource<Int, PostEntity> {
        return UserDatabase.get().globalFeedV5PagingDao().queryGlobalFeedPost()
    }

    fun incrementCommentCount(postId: String) {
        UserDatabase.get().postDao().incrementCommentCount(postId)
    }

    fun decrementCommentCount(postId: String) {
        UserDatabase.get().postDao().decrementCommentCount(postId)
    }
    
    fun getPostByIds(postIds: List<String>) : Flowable<List<PostEntity>> {
        return UserDatabase.get().postDao().getByPostIds(postIds)
    }

    fun notifyPost(postId: String) {
        UserDatabase.get().postDao().updatePost(postId)
    }

    fun getLatestPosts(
        targetId: String,
        targetType: String,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        includeDeleted: Boolean?,
        postTypes: List<String>, feedType: AmityFeedType?
    ): Flowable<PostEntity> {
        return UserDatabase.get().postDao()
            .getLatestPost(
                targetId, targetType, includeDeleted, postTypes,
                dynamicQueryStreamKeyCreator.toMap().hashCode(), nonce, feedType
            )
    }
}
