package com.amity.socialcloud.sdk.social.data.post

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.tombstone.TombstoneRepository
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.social.feed.AmityCommunityFeedSortOption
import com.amity.socialcloud.sdk.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.social.feed.AmityPost
import com.amity.socialcloud.sdk.social.feed.AmityUserFeedSortOption
import com.ekoapp.ekosdk.AmityObjectRepository
import com.ekoapp.ekosdk.internal.AmityPostAttachment
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.TombstoneModelType
import com.ekoapp.ekosdk.internal.data.boundarycallback.CommunityFeedBoundaryCallback
import com.ekoapp.ekosdk.internal.data.boundarycallback.GlobalFeedBoundaryCallback
import com.ekoapp.ekosdk.internal.data.boundarycallback.UserFeedBoundaryCallback
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.post.CommunityPostMediator
import com.ekoapp.ekosdk.internal.repository.post.CustomPostRankingMediator
import com.ekoapp.ekosdk.internal.repository.post.GlobalFeedMediator
import com.ekoapp.ekosdk.internal.repository.post.UserPostMediator
import com.ekoapp.ekosdk.internal.repository.post.helper.PostComposerFunction
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class PostRepository : AmityObjectRepository<PostEntity, AmityPost>() {

    override fun fetchAndSave(objectId: String): Completable {
        return PostRemoteDataStore().fetchPost(objectId)
            .flatMapCompletable { PostQueryPersister().persist(it) }
            .onErrorResumeNext {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    PostLocalDataStore().hardDelete(objectId).andThen(Completable.error(it))
                } else {
                    Completable.error(it)
                }
            }
    }

    override fun queryFromCache(objectId: String): PostEntity? {
        return PostLocalDataStore().getPost(objectId)
    }

    override fun mapper(): ModelMapper<PostEntity, AmityPost> {
        return PostModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<PostEntity> {
        return PostLocalDataStore().observe(objectId)
    }

    fun getPostByIds(postIds: List<String>): Flowable<List<AmityPost>> {
        return PostRemoteDataStore().getPostByIds(postIds)
                .flatMap {
                    PostQueryPersister().persist(it)
                            .andThen(Single.just((it.posts.map { it.postId })))
                }
                .flatMapPublisher { ids ->
                    PostLocalDataStore().getPostByIds(ids)
                            .map { posts ->
                                posts.map {
                                    PostModelMapper().map(it)
                                }
                            }
                }
    }

    @Deprecated("pagedList will be unsupported soon")
    fun getGlobalFeedPagedList(): io.reactivex.Flowable<PagedList<AmityPost>> {
        val factory = PostLocalDataStore().getGlobalPostsDataSourceFactory()
        val boundaryCallback = GlobalFeedBoundaryCallback(getDefaultPageSize())
        return createRxCollectionWithBoundaryCallback(
            factory = factory.map(PostComposerFunction()),
            callback = boundaryCallback
        )
    }

    @Deprecated("pagedList will be unsupported soon")
    fun getUserFeedPagedList(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?
    ): io.reactivex.Flowable<PagedList<AmityPost>> {
        val factory =
            PostLocalDataStore().getUserPostsDataSourceFactory(userId, sortOption, isDeleted)
        val boundaryCallback = UserFeedBoundaryCallback(
            userId = userId,
            sortOption = sortOption.apiKey,
            isDeleted = isDeleted,
            pageSize = getDefaultPageSize()
        )
        return createRxCollectionWithBoundaryCallback(
            factory.map(PostComposerFunction()),
            boundaryCallback
        )
    }

    fun getUserPostPagingData(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf()
    ): io.reactivex.Flowable<PagingData<AmityPost>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = UserPostMediator(
                userId = userId,
                sortBy = sortOption.apiKey,
                isDeleted = isDeleted,
                postTypes = postTypes
            ),
            pagingSourceFactory = {
                PostLocalDataStore()
                    .getUserPostsPagingSource(userId, sortOption, isDeleted, postTypes)
            },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create()
    }

    @Deprecated("pagedList will be unsupported soon")
    fun getCommunityFeedPagedList(
        communityId: String,
        sortOption: AmityCommunityFeedSortOption,
        isDeleted: Boolean?,
        feedType: AmityFeedType
    ): io.reactivex.Flowable<PagedList<AmityPost>> {
        val factory = PostLocalDataStore().getCommunityPostsDataSourceFactory(
            communityId,
            sortOption,
            isDeleted,
            feedType
        )
        val boundaryCallback = CommunityFeedBoundaryCallback(
            communityId = communityId,
            sortOption = sortOption.apiKey,
            isDeleted = isDeleted,
            feedType = feedType.apiKey,
            pageSize = getDefaultPageSize()
        )
        return createRxCollectionWithBoundaryCallback(
            factory.map(PostComposerFunction()),
            boundaryCallback
        )
    }

    fun getCommunityPostPagingData(
        communityId: String,
        sortOption: AmityCommunityFeedSortOption,
        feedType: AmityFeedType,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf()
    ): io.reactivex.Flowable<PagingData<AmityPost>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = CommunityPostMediator(
                communityId = communityId,
                sortBy = sortOption.apiKey,
                isDeleted = isDeleted,
                feedType = feedType,
                postTypes = postTypes
            ),
            pagingSourceFactory = {
                PostLocalDataStore().getCommunityPostsPagingSource(
                    communityId,
                    sortOption,
                    feedType,
                    isDeleted,
                    postTypes
                )
            },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create()
    }

    fun getGlobalFeedPagingData(): io.reactivex.Flowable<PagingData<AmityPost>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            queryStreamMediator = GlobalFeedMediator(),
            pagingSourceFactory = { PostLocalDataStore().getGlobalPostsPagingSource() },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create()

    }

    fun getCustomPostRankingPagingData(): io.reactivex.Flowable<PagingData<AmityPost>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            queryStreamMediator = CustomPostRankingMediator(),
            pagingSourceFactory = { PostLocalDataStore().getGlobalPostsV5PagingSource() },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create()
    }

    fun getPost(postId: String): AmityPost? {
        val post = PostLocalDataStore().getPost(postId)
        return if (post == null) null else PostModelMapper().map(post)
    }

    fun getPosts(postIds: List<String>): List<AmityPost> {
        val posts = if (postIds.isNotEmpty()) {
            PostLocalDataStore().getPosts(postIds).map {
                PostModelMapper().map(it)
            }
        } else {
            emptyList()
        }
        return posts
    }

    fun createPostV4(
        targetType: String, targetId: String,
        data: JsonObject, attachments: List<AmityPostAttachment>?,
        dataType: AmityPost.DataType? = null,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): Single<String> {
        return PostRemoteDataStore().createPost(
            targetType,
            targetId,
            data,
            attachments,
            dataType,
            metadata,
            mentionees
        )
            .flatMap {
                PostQueryPersister().persist(it)
                    .andThen(Single.just(it.posts.first()?.postId ?: ""))
            }
    }

    fun deletePost(postId: String, hardDelete: Boolean): Completable {
        return PostRemoteDataStore().deletePost(postId, hardDelete)
            .flatMapCompletable {
                if (!it.isSuccess) {
                    Completable.complete()
                } else if (hardDelete) {
                    PostLocalDataStore().hardDelete(postId).andThen(Completable.defer {
                        TombstoneRepository().saveTombstone(
                            objectId = postId,
                            tombstoneModelType = TombstoneModelType.POST,
                            amityError = AmityError.ITEM_NOT_FOUND
                        )
                    })
                } else {
                    PostLocalDataStore().softDelete(postId)
                }
            }
    }

    fun approvePost(postId: String): Completable {
        var initialFeedType: AmityFeedType = AmityFeedType.NONE
        return Flowable.fromCallable {
            initialFeedType = PostLocalDataStore().getFeedType(postId)
            PostLocalDataStore().updateFeedType(postId, AmityFeedType.PUBLISHED)
        }.flatMapCompletable {
            PostRemoteDataStore().approvePost(postId).ignoreElement()
        }.doOnError {
            if (AmityError.from(it) == AmityError.FORBIDDEN_ERROR) {
                PostLocalDataStore().updateFeedType(postId, AmityFeedType.NONE)
            } else {
                PostLocalDataStore().updateFeedType(postId, initialFeedType)
            }
        }
    }

    fun declinePost(postId: String): Completable {
        var initialFeedType: AmityFeedType = AmityFeedType.NONE
        return Flowable.fromCallable {
            initialFeedType = PostLocalDataStore().getFeedType(postId)
            PostLocalDataStore().updateFeedType(postId, AmityFeedType.DECLINED)
        }.flatMapCompletable {
            PostRemoteDataStore().declinePost(postId).ignoreElement()
        }.doOnError {
            if (AmityError.from(it) == AmityError.FORBIDDEN_ERROR) {
                PostLocalDataStore().updateFeedType(postId, AmityFeedType.NONE)
            } else {
                PostLocalDataStore().updateFeedType(postId, initialFeedType)
            }
        }
    }

    fun editPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?,
        attachments: List<AmityPostAttachment>?
    ): Completable {
        return PostRemoteDataStore().editPost(postId, data, metadata, mentionees, attachments)
            .flatMapCompletable {
                PostQueryPersister().persist(it)
            }
    }

    fun flagPost(postId: String): Completable {
        return PostRemoteDataStore().flagPost(postId)
            .flatMapCompletable {
                PostQueryPersister().persist(it)
            }
    }

    fun unFlagPost(postId: String): Completable {
        return PostRemoteDataStore().unFlagPost(postId)
            .flatMapCompletable {
                PostQueryPersister().persist(it)
            }
    }

    fun getLatestPost(
        targetId: String,
        targetType: String,
        includeDeleted: Boolean?,
        postTypes: List<AmityPost.DataType>,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        feedType: AmityFeedType? = null
    ): Flowable<AmityPost> {
        return PostLocalDataStore().getLatestPosts(
            targetId = targetId,
            targetType = targetType,
            includeDeleted = includeDeleted,
            postTypes = postTypes.map { it.getApiKey() },
            feedType = feedType,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).map {
            PostModelMapper().map(it)
        }
    }

    fun isFlaggedByMe(postId: String): Single<JsonObject> {
        return PostRemoteDataStore().isFlaggedByMe(postId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }


}