package com.amity.socialcloud.sdk.social.domain.post

import com.amity.socialcloud.sdk.core.domain.ComposerUseCase
import com.amity.socialcloud.sdk.core.domain.file.FileGetUseCase
import com.amity.socialcloud.sdk.core.domain.file.ImageGetUseCase
import com.amity.socialcloud.sdk.social.domain.post.flag.IsPostFlaggedByMeUseCase
import com.amity.socialcloud.sdk.core.domain.reaction.MyReactionsGetUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetUseCase
import com.amity.socialcloud.sdk.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.core.reaction.ReactionReferenceType
import com.amity.socialcloud.sdk.social.domain.comment.LatestCommentsGetUseCase
import com.amity.socialcloud.sdk.social.domain.community.CommunityGetUseCase
import com.amity.socialcloud.sdk.social.domain.community.member.CommunityMemberGetUseCase
import com.amity.socialcloud.sdk.social.feed.AmityPost

internal class PostComposerUseCase : ComposerUseCase() {

    fun execute(post: AmityPost): AmityPost {
        addChildren(post)
        addData(post)
        addMyReactions(post)
        addMyFlag(post)
        addPostedUser(post)
        addSharedUser(post)
        //addTarget(post)
        addLatestComments(post)
        addMentionees(post)
        return post
    }

    private fun addChildren(post: AmityPost) {
        post.children = PostsGetUseCase().execute(post.childrenPostIds)
    }

    private fun addData(post: AmityPost) {
        when (post.data) {
            is AmityPost.Data.IMAGE -> {
                post.data.image = ImageGetUseCase().execute(post.data.fileId)
            }
            is AmityPost.Data.FILE -> {
                post.data.file = FileGetUseCase().execute(post.data.fileId)
            }
            is AmityPost.Data.VIDEO -> {
                post.data.thumbnail = ImageGetUseCase().execute(post.data.thumbnailFileId)
            }
            else -> {
                // do nothing
            }
        }
    }

    private fun addMyReactions(post: AmityPost) {
        post.myReactions =
            MyReactionsGetUseCase().execute(ReactionReferenceType.POST, post.getPostId())
    }

    private fun addMyFlag(post: AmityPost) {
        post.isFlaggedByMe = IsPostFlaggedByMeUseCase().execute(post.getPostId())
    }

    private fun addPostedUser(post: AmityPost) {
        post.postedUser = UserGetUseCase().execute(post.getPostedUserId())
    }

    private fun addSharedUser(post: AmityPost) {
        post.sharedUserId?.let {
            post.sharedUser = UserGetUseCase().execute(it)
        }
    }

    private fun addLatestComments(post: AmityPost) {
        post.latestComments = LatestCommentsGetUseCase().execute("post", post.getPostId())
    }

    private fun addMentionees(post: AmityPost) {
        post.mentionees = post.getMentionees().map {
            if (it is AmityMentionee.USER) {
                it.user = UserGetUseCase().execute(it.getUserId())
            }
            it
        }
    }

}