package com.amity.socialcloud.sdk.video.data.stream

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.video.stream.AmityStream
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.EkoStreamQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.EkoStreamEntity
import com.ekoapp.ekosdk.internal.keycreator.StreamKeyCreator
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.stream.StreamReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@ExperimentalPagingApi
internal class StreamMediator(
    val statuses: List<String>
) : DynamicQueryStreamMediator<EkoStreamEntity, EkoStreamQueryDto, AmityStream>(
        nonce = AmityNonce.STREAM_LIST,
        dynamicQueryStreamKeyCreator = StreamKeyCreator(statuses),
) {
    
    override fun provideReactorPublisher(): Flowable<AmityStream> {
        return StreamReactorUseCase().execute(statuses).toV2()
    }
    
    override fun forceRefresh(): Boolean = true
    
    override fun getFirstPageRequest(pageSize: Int): Single<EkoStreamQueryDto> {
        return getRequest(limit = pageSize)
    }
    
    override fun getFetchByTokenRequest(token: String): Single<EkoStreamQueryDto> {
        return getRequest(token = token)
    }
    
    override fun persistResponse(dto: EkoStreamQueryDto): Completable {
        return StreamListQueryPersister().persist(dto).toV2()
    }
    
    override fun convertResponseToQueryToken(dto: EkoStreamQueryDto): QueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
                dynamicQueryStreamKeyCreator = StreamKeyCreator(statuses),
                next = dto.token.next,
                previous = dto.token?.previous,
                primaryKeys = dto.result.streams.map { post -> post.streamId }
        )
    }
    
    private fun getRequest(
            limit: Int? = null,
            token: String? = null
    ): Single<EkoStreamQueryDto> {
        return StreamRemoteDataStore().queryVideoStreaming(
            isLive = null,
            statuses = statuses,
            userPublicIds = null,
            sortBy = null,
            isDeleted = null,
            limit = limit,
            token = token
        ).toV2()
    }
}