package com.amity.socialcloud.sdk.video.data.stream

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagedList
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.video.model.AmityBroadcastResolution
import com.amity.socialcloud.sdk.video.stream.AmityStream
import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.socket.request.CreateStreamRequest
import com.ekoapp.ekosdk.internal.data.boundarycallback.EkoStreamBoundaryCallback
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.subjects.PublishSubject

internal class StreamRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun getStreamCollection(statuses: Array<AmityStream.Status>): Flowable<PagedList<AmityStream>> {
        val statusArray = getFormattedStatuses(statuses)
        val factory = StreamLocalDataStore().getStreams(statusArray)
            .map { StreamRepositoryHelper().attachDataToEntity(it) }
            .map { StreamModelMapper().map(it) }

        val delaySubject = PublishSubject.create<Boolean>()
        val boundaryCallback = EkoStreamBoundaryCallback(
            statuses = statusArray,
            isReconnecting = false,
            pageSize = getDefaultPageSize(),
            delaySubject = delaySubject
        )
        return createRxCollectionWithBoundaryCallback(
            factory.map(boundaryCallback),
            boundaryCallback
        ).toV3()
    }
    
    @OptIn(ExperimentalPagingApi::class)
    fun getStreamCollectionPagingData(statuses: Array<AmityStream.Status>): Flowable<PagingData<AmityStream>> {
        val statusList = getFormattedStatuses(statuses).toList()
        val pagerCreator = DynamicQueryStreamPagerCreator(
                pagingConfig = PagingConfig(
                        pageSize = getDefaultPageSize(),
                        enablePlaceholders = false
                ),
                dynamicQueryStreamMediator = StreamMediator(
                    statusList
                ),
                pagingSourceFactory = {
                    StreamLocalDataStore().getStreamsPagingSource(statusList)
                },
                modelMapper = StreamModelMapper()
        )
        return pagerCreator.create().toV3()
    }

    fun observeStream(streamId: String): Flowable<AmityStream> {
        return StreamLocalDataStore().observeStream(streamId)
            .map {
                StreamModelMapper().map(it)
            }
    }

    fun fetchStream(streamId: String): Completable {
        return StreamRemoteDataStore().getVideoStreaming(streamId)
            .flatMapCompletable {
                StreamQueryPersister().persist(it)
            }
    }

    fun createStream(
        title: String,
        description: String,
        resolution: AmityBroadcastResolution,
        thumbnailFileId: String? = null
    ): Single<AmityStream> {
        val request = CreateStreamRequest(
            title = title,
            description = description,
            resolution = resolution.readableName,
            thumbnailFileId = thumbnailFileId
        )

        return StreamRemoteDataStore().createVideoStreaming(request)
            .flatMap {
                StreamQueryPersister().persist(it)
                    .andThen(Single.just(it.streams.first().streamId))
            }.map {
                getStream(it)
            }
    }

    fun createStreamBroadcaster(
        title: String,
        description: String,
        resolution: AmityBroadcastResolution,
        thumbnailFileId: String? = null
    ): Single<Pair<String, String?>> {
        return createStream(title, description, resolution, thumbnailFileId)
            .map {
                val streamId: String = it.getStreamId()
                val broadcastingUrl: String? = it.getBroadcasterData()?.getUrl()

                (streamId to broadcastingUrl)
            }
    }

    fun disposeVideoStreaming(streamId: String): Completable {
        return StreamRemoteDataStore().disposeVideoStreaming(streamId).ignoreElement()
    }

    private fun getFormattedStatuses(statuses: Array<AmityStream.Status>): Array<String> {
        val statusSet = if (statuses.isNotEmpty()) {
            statuses.toSet()
        } else {
            setOf(AmityStream.Status.IDLE)
        }
        return statusSet.map(AmityStream.Status::apiKey).toTypedArray()
    }

    private fun getStream(streamId: String): AmityStream? {
        return StreamLocalDataStore().getStream(streamId)?.let {
            StreamRepositoryHelper().attachDataToEntity(it)
        }?.let {
            StreamModelMapper().map(it)
        }
    }
    
    fun getLatestSteam(statuses: List<String>): Flowable<AmityStream> {
        return StreamLocalDataStore().getLatestStream(statuses = statuses)
                .map(StreamModelMapper()::map)
    }
}