package com.ekoapp.ekosdk


import androidx.paging.DataSource
import androidx.paging.PagedList
import androidx.paging.PagedList.BoundaryCallback
import androidx.paging.RxPagedListBuilder
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.amity.socialcloud.sdk.core.exception.EntityExpiredException
import com.amity.socialcloud.sdk.core.exception.EntityNotFoundException
import com.amity.socialcloud.sdk.log.AmityLog
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal abstract class AmityObjectRepository<Entity: EkoObject, PublicModel: Any> {

    companion object {
        const val INITIAL_LOAD_KEY_VALUE = 0
        const val DEFAULT_PAGE_SIZE = 20
    }

    @Deprecated("Already moved to paging data, delete once breaking change is possible")
    private fun <T : Any> createRxCollectionBuilder(factory: DataSource.Factory<Int, T>, enablePlaceholders: Boolean)
            : RxPagedListBuilder<Int, T> {
        val config = PagedList.Config.Builder()
                .setPageSize(DEFAULT_PAGE_SIZE)
                .setEnablePlaceholders(enablePlaceholders)
                .build()
        return RxPagedListBuilder(factory, config)
    }

    @Deprecated("Already moved to paging data, delete once breaking change is possible")
    fun <T : Any> createRxCollectionWithBoundaryCallback(factory: DataSource.Factory<Int, T>,
                                                         callback: BoundaryCallback<T>): Flowable<PagedList<T>> {
        return createRxCollectionWithBoundaryCallback(factory, callback, INITIAL_LOAD_KEY_VALUE)
    }

    @Deprecated("Already moved to paging data, delete once breaking change is possible")
    fun <T : Any> createRxCollectionWithBoundaryCallback(factory: DataSource.Factory<Int, T>,
                                                         callback: BoundaryCallback<T>,
                                                         initialLoadKey: Int): Flowable<PagedList<T>> {
        return createRxCollectionWithBoundaryCallback(factory, callback, initialLoadKey, false)
    }

    @Deprecated("Already moved to paging data, delete once breaking change is possible")
    fun <T : Any> createRxCollectionWithBoundaryCallback(factory: DataSource.Factory<Int, T>,
                                                         callback: BoundaryCallback<T>,
                                                         initialLoadKey: Int,
                                                         enablePlaceholders: Boolean): Flowable<PagedList<T>> {
        return createRxCollectionBuilder(factory, enablePlaceholders)
                .setBoundaryCallback(callback)
                .setInitialLoadKey(initialLoadKey)
                .buildFlowable(BackpressureStrategy.BUFFER)
    }

    abstract fun fetchAndSave(objectId: String): Completable

    abstract fun queryFromCache(objectId: String): Entity?

    abstract fun mapper(): ModelMapper<Entity, PublicModel>

    abstract fun observeFromCache(objectId: String): io.reactivex.rxjava3.core.Flowable<Entity>

    open fun observe(objectId: String): io.reactivex.rxjava3.core.Flowable<PublicModel>{
        return observeFromCache(objectId)
            .map { mapper().map(it) }
    }

    open fun obtain(objectId: String): Single<PublicModel> {
        val TAG = "OTP2"
        val queryFromCacheSingle = Single.fromCallable {
            //TODO Remove all logs once tested
            AmityLog.tag(TAG).e("query cached data")
            return@fromCallable queryFromCache(objectId) ?: throw EntityNotFoundException
        }
        return queryFromCacheSingle
            .subscribeOn(Schedulers.io())
            .flatMap {
                val isNotExpired = it.expiresAt?.isAfterNow == true
                if (isNotExpired) {
                    AmityLog.tag(TAG).e("return cached data")
                    return@flatMap Single.just(it)
                } else {
                    return@flatMap Single.error(EntityExpiredException)
                }
            }
            .onErrorResumeNext {
                AmityLog.tag(TAG).e("${it.message}")
                if (it == EntityNotFoundException || it == EntityExpiredException) {
                    AmityLog.tag(TAG).e("fetching object from server")
                    return@onErrorResumeNext fetchAndSave(objectId)
                            .andThen(Single.defer { queryFromCacheSingle })
                            .onErrorResumeNext { error ->
                                Single.error(AmityException.create("unknown error", error, AmityError.UNKNOWN))
                            }
                } else {
                    return@onErrorResumeNext Single.error(it)
                }
            }.map { mapper().map(it) }
    }
}