package com.ekoapp.ekosdk;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.MINUTES;

import android.util.LruCache;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MediatorLiveData;
import androidx.lifecycle.MutableLiveData;

import com.amity.socialcloud.sdk.log.AmityLog;
import com.amity.socialcloud.sdk.socket.model.SocketRequest;
import com.amity.socialcloud.sdk.socket.util.EkoGson;
import com.ekoapp.ekosdk.internal.api.EkoSocket;
import com.ekoapp.ekosdk.internal.api.SocketConnectionEvent;
import com.ekoapp.ekosdk.internal.api.socket.call.Call;

import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.schedulers.Schedulers;

public class EkoLiveData<T> extends MediatorLiveData<T> {

    public enum DataStatus {
        NOT_EXIST,
        LOCAL,
        FRESH,
        ERROR;
    }

    public enum LoadingStatus {
        NOT_LOADING,
        LOADING,
        LOADED,
        ERROR;
    }

    private static final String TAG = EkoLiveData.class.getName();
    private static final long LIVE_QUERY_EXPIRY = MILLISECONDS.convert(5, MINUTES);

    private final MutableLiveData<DataStatus> dataStatus = new MutableLiveData<>();
    private final MutableLiveData<LoadingStatus> loadingStatus = new MutableLiveData<>();

    private final CompositeDisposable disposable = new CompositeDisposable();
    @Nullable
    private final Call<?> liveQueryCall;

    private static LruCache<SocketRequest, Long> lastQueryTimeMap = new LruCache<SocketRequest, Long>(100) {
        @Override
        protected Long create(SocketRequest request) {
            return 0L;
        }
    };

    private EkoLiveData(@NonNull LiveData<T> source, @NonNull T proxy, @Nullable Call liveQueryCall) {
        this.liveQueryCall = liveQueryCall;
        dataStatus.postValue(DataStatus.NOT_EXIST);
        loadingStatus.postValue(LoadingStatus.NOT_LOADING);
        addSource(source, item -> {
            if (item != null) {
                postValue(item);
                dataStatus.postValue(shouldCallLiveQuery(liveQueryCall) ? DataStatus.LOCAL : DataStatus.FRESH);
            } else {
                //postValue(proxy);
                dataStatus.postValue(DataStatus.NOT_EXIST);
            }
        });
    }

    public LiveData<DataStatus> getDataStatus() {
        return dataStatus;
    }

    public LiveData<LoadingStatus> getLoadingStatus() {
        return loadingStatus;
    }

    private static boolean shouldCallLiveQuery(@Nullable Call<?> call) {
        if (call == null) {
            return false;
        }
        SocketRequest request = call.getRequest();
        final long lastSuccessQueryTime = lastQueryTimeMap.get(request);
        AmityLog.INSTANCE.INSTANCE.tag(TAG).d("request: %s lastSuccessQueryTime: %s", EkoGson.get().toJson(request), lastSuccessQueryTime);
        AmityLog.INSTANCE.tag(TAG).d("lastQueryTimeMap: %s size: %s", lastQueryTimeMap, lastQueryTimeMap.size());
        long now = System.currentTimeMillis();
        long timeSinceLastQuery = now - lastQueryTimeMap.get(request);
        return timeSinceLastQuery > LIVE_QUERY_EXPIRY;
    }

    private void callLiveQueryIfNeeded(SocketConnectionEvent event, Call<?> call) {
        SocketRequest request = call.getRequest();
        if (shouldCallLiveQuery(call)) {
            if (event.isConnected()) {
                final long lastSuccessQueryTime = lastQueryTimeMap.get(request);
                AmityLog.INSTANCE.tag(TAG).i("restart live query: %s", request);
                EkoSocket.call(call)
                        .doOnSubscribe(disposable -> {
                            lastQueryTimeMap.put(request, System.currentTimeMillis());
                            loadingStatus.postValue(LoadingStatus.LOADING);
                        })
                        .doOnSuccess(ignored -> {
                            lastQueryTimeMap.put(request, System.currentTimeMillis());
                            loadingStatus.postValue(LoadingStatus.LOADED);
                        })
                        .doOnError(e -> {
                            lastQueryTimeMap.put(request, lastSuccessQueryTime);
                            loadingStatus.postValue(LoadingStatus.ERROR);
                        })
                        .subscribe();
            } else {
                AmityLog.INSTANCE.tag(TAG).e("live query expired but was not connected to socket");
            }
        }
    }

    @Override
    protected void onActive() {
        super.onActive();
        Call<?> call = liveQueryCall;
        if (call != null) {
            disposable.add(EkoSocket
                    .connectionEvent()
                    .subscribeOn(Schedulers.io())
                    .subscribe(event -> callLiveQueryIfNeeded(event, call)));
        }
    }

    @Override
    protected void onInactive() {
        super.onInactive();
        disposable.clear();
    }


    static <T> EkoLiveData<T> create(@NonNull LiveData<T> source, @NonNull T proxy, @Nullable Call liveQueryCall) {
        return new EkoLiveData<>(source, proxy, liveQueryCall);
    }
}
