package com.amity.socialcloud.sdk.chat.channel

import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityException
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.usecase.channel.CreateChannelUseCase
import com.ekoapp.ekosdk.internal.usecase.channel.CreateChannelWithChannelIdUseCase
import com.ekoapp.ekosdk.internal.usecase.channel.CreateConversationWithUserIdsUseCase
import io.reactivex.Single

class AmityChannelCreator {
    private var channelType: CreationType? = null
    private var option = ChannelCreateOption()
    private var channelId: String? = null

    internal constructor(channelType: CreationType, option: ChannelCreateOption) {
        this.channelType = channelType
        this.option = option
    }

    internal constructor(channelType: CreationType, channelId: String?, option: ChannelCreateOption) {
        this.channelType = channelType
        this.channelId = channelId
        this.option = option
    }

    fun create(): Single<AmityChannel> {
        return if (channelType != null) {
            when (channelType) {
                CreationType.LIVE -> {
                    createChannelByType(channelId, CreationType.LIVE, option)
                }
                CreationType.PRIVATE -> {
                    createChannelByType(channelId, CreationType.PRIVATE, option)
                }
                CreationType.COMMUNITY -> {
                    createChannelByType(channelId, CreationType.COMMUNITY, option)
                }
                CreationType.STANDARD -> {
                    createChannelByType(channelId, CreationType.STANDARD, option)
                }
                CreationType.CONVERSATION -> {
                    createChannelConversation(option)
                }

                else -> {
                    Single.error(AmityException.create("this channel type is not supported", null, AmityError.INVALID_PARAMETER))
                }
            }
        } else {
            Single.error(AmityException.create("channel type must not be null", null, AmityError.INVALID_PARAMETER))
        }
    }

    private fun createChannelByType(channelId: String?, channelType: CreationType,
                                    option: ChannelCreateOption): Single<AmityChannel> {
        return if (channelId.isNullOrEmpty()) {
            createChannel(channelType, option)
        } else {
            createChannel(channelId, channelType, option)
        }

    }

    private fun createChannelConversation(option: ChannelCreateOption): Single<AmityChannel> {
        return CreateConversationWithUserIdsUseCase().execute(option).toV2()
    }

    private fun createChannel(channelId: String,
                              creationType: CreationType,
                              option: ChannelCreateOption): Single<AmityChannel> {
        return CreateChannelWithChannelIdUseCase().execute(channelId, creationType, option).toV2()
    }

    private fun createChannel(creationType: CreationType,
                              option: ChannelCreateOption): Single<AmityChannel> {
        return CreateChannelUseCase().execute(creationType, option).toV2()
    }

    internal enum class CreationType(val apiKey: String) {
        COMMUNITY(AmityChannel.Type.COMMUNITY.apiKey),
        STANDARD(AmityChannel.Type.STANDARD.apiKey),
        LIVE(AmityChannel.Type.LIVE.apiKey),
        PRIVATE(AmityChannel.Type.PRIVATE.apiKey),
        CONVERSATION(AmityChannel.Type.CONVERSATION.apiKey);

        companion object {
            fun fromJson(jsonValue: String): CreationType {
                run breaker@{
                    for (type in values()) {
                        if (type.apiKey == jsonValue) {
                            return@breaker type
                        }
                    }
                }
                return COMMUNITY
            }
        }
    }

}