package com.amity.socialcloud.sdk.social.comment

import android.os.Parcelable
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.core.events.AmityTopic
import com.amity.socialcloud.sdk.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.core.events.comment.AmityCommentEvents
import com.amity.socialcloud.sdk.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.core.reaction.AmityReactor
import com.amity.socialcloud.sdk.core.reaction.ReactionReferenceType
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.core.utils.requiredNonNull
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.data.model.AmityReactionMap
import com.ekoapp.ekosdk.internal.usecase.comment.DeleteCommentUseCase
import com.google.common.base.Objects
import com.google.gson.JsonObject
import io.reactivex.Completable
import kotlinx.android.parcel.Parcelize
import kotlinx.android.parcel.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject, JsonObjectParceler>
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityComment internal constructor(
        private var commentId: String = ObjectId.get().toHexString(),
        internal var referenceType: String,
        internal var referenceId: String,
        private var userId: String,
        private var parentId: String?,
        private var rootId: String?,
        private var dataType: String = DataType.TEXT.apiKey,
        private var data: JsonObject,
        private var metadata: JsonObject?, // Remain private until further requirement change
        private var childrenNumber: Int,
        internal var latestReplies: List<AmityComment> = emptyList(),
        private var flagCount: Int,
        internal var myReactions: List<String>,
        private var reactionCount: Int,
        private var reactions: AmityReactionMap,
        private var isDeleted: Boolean,
        private var editedAt: DateTime?,
        private var createdAt: DateTime?,
        private var updatedAt: DateTime?,
        private var syncState: String,
        internal var mentionees: List<AmityMentionee>,
        internal var user: AmityUser?,
        internal var isFlaggedByMe: Boolean = false,
        internal val path: String
        ) : Parcelable, ReactorObject {

    fun getCommentId(): String {
        return commentId
    }

    fun getReference(): Reference {
        return Reference.from(referenceType, referenceId)
    }

    fun getParentId(): String? {
        return parentId
    }

    fun getUserId(): String {
        return userId
    }

    fun getState(): State {
        return State.enumOf(syncState)
    }

    fun getDataType(): DataType {
        return DataType.enumOf(dataType)
    }

    fun getData(): Data {
        return Data.from(getDataType(), data)
    }

    fun getChildrenNumber(): Int {
        return childrenNumber
    }

    fun getFlagCount(): Int {
        return flagCount
    }

    fun getReactionCount(): Int {
        return reactionCount
    }

    fun getReactionMap(): AmityReactionMap {
        return reactions
    }

    fun getMyReactions(): List<String> {
        return myReactions
    }

    fun getLatestReplies(): List<AmityComment> {
        return latestReplies
    }

    fun getReactions(): AmityCommentReactionQuery.Builder {
        return AmityCommentReactionQuery.Builder(commentId)
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getEditedAt(): DateTime? {
        return editedAt
    }

    fun getCreatedAt(): DateTime? {
        return createdAt
    }

    fun getUpdatedAt(): DateTime? {
        return updatedAt
    }

    fun getUser(): AmityUser? {
        return user
    }

    fun isEdited(): Boolean {
        return requiredNonNull(editedAt?.isAfter(createdAt))
    }

    fun isFlaggedByMe(): Boolean {
        return isFlaggedByMe
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMentionees(): List<AmityMentionee> {
        return mentionees
    }

    fun delete(): Completable {
        return delete(false)
    }

    fun delete(hardDelete: Boolean): Completable {
        return DeleteCommentUseCase().execute(commentId, hardDelete).toV2()
    }

    fun react(): AmityReactor {
        return AmityReactor(ReactionReferenceType.COMMENT, commentId)
    }

    fun report(): AmityCommentFlagger {
        return AmityCommentFlagger(commentId)
    }

    sealed class Reference{

        class POST(private val postId: String) : Reference() {
            fun getPostId(): String {
                return postId
            }
        }

        class CONTENT(private val contentId: String) : Reference() {
            fun getContentId(): String {
                return contentId
            }
        }

        object UNKNOWN : Reference()

        companion object {
            internal fun from(referenceType: String, referenceId: String): Reference {
                return when (referenceType) {
                    "post" -> {
                        POST(referenceId)
                    }
                    "content" -> {
                        CONTENT(referenceId)
                    }
                    else -> {
                        UNKNOWN
                    }
                }
            }
        }
    }

    sealed class Data : Parcelable {

        @Parcelize
        class TEXT(private val commentId: String,
                   private val text: String? = "") : Data() {

            fun getCommentId(): String {
                return commentId
            }

            fun getText(): String {
                return text ?: ""
            }

            fun edit(): AmityTextCommentEditOption {
                return AmityTextCommentEditOption((getCommentId()))
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is TEXT
                        && Objects.equal(other.commentId , commentId)
                        && Objects.equal(other.text , text))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(commentId, text)
            }
        }

        companion object {
            internal fun from(dataType: DataType, data: JsonObject): Data {
                return EkoGson.get().fromJson(data, TEXT::class.java)
            }
        }

    }

    enum class DataType(val apiKey: String) {
        TEXT("text");

        companion object {
            fun enumOf(value: String): DataType {
                return values().find { it.apiKey == value } ?: TEXT
            }
        }
    }

    enum class State(val stateName: String) {
        CREATED("created"),
        SYNCING("syncing"),
        SYNCED("synced"),
        FAILED("failed");

        companion object {
            fun enumOf(value: String): State = values().find { it.stateName == value } ?: SYNCED
        }
    }

    fun subscription(events: AmityCommentEvents): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.COMMENT(this, events))
    }

    override fun uniqueId(): String {
        return commentId
    }

    override fun updatedAt(): DateTime? {
        return updatedAt
    }
}