package com.amity.socialcloud.sdk.social.comment

import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.usecase.comment.CommentManualFirstPageLoadUseCase
import com.ekoapp.ekosdk.internal.usecase.comment.CommentManualLoadWithTokenUsecase
import com.ekoapp.ekosdk.internal.usecase.comment.CommentManualQueryUseCase
import io.reactivex.rxjava3.core.BackpressureStrategy
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.rxjava3.subjects.BehaviorSubject

private const val COMMENT_LOADER_PAGE_SIZE = 15

class AmityCommentLoader internal constructor(
    private val referenceId: String,
    private val referenceType: String,
    private val isFilterByParentId: Boolean,
    private val parentId: String?,
    private val isDeleted: Boolean?,
    private val sortOption: AmityCommentSortOption
) {

    private var ids = mutableListOf<String>()
    private val idPublisher = BehaviorSubject.create<List<String>>()

    private var token: String? = null
    private var isLoading = false

    fun hasMore(): Boolean {
        return token?.isNotEmpty() ?: true
    }

    fun getResult(): io.reactivex.Flowable<List<AmityComment>> {
        return getResultV3().toV2()
    }

    private fun getResultV3(): Flowable<List<AmityComment>> {
        return idPublisher
            .toFlowable(BackpressureStrategy.BUFFER)
            .switchMap {
                CommentManualQueryUseCase().execute(
                    referenceType,
                    referenceId,
                    isFilterByParentId,
                    parentId,
                    isDeleted,
                    sortOption,
                    ids,
                    hasMore()
                )
            }.map { list ->
                ids = list.map { it.getCommentId() }.toMutableList()
                list
            }.doOnSubscribe {
                idPublisher.onNext(ids)
            }
    }

    fun load(): io.reactivex.Completable {
        return makeRequest().toV2()
    }

    private fun makeRequest(): Completable {
        if (isLoading) return Completable.never()
        isLoading = true
        return when {
            token == null -> {
                // first load
                CommentManualFirstPageLoadUseCase().execute(
                    referenceId,
                    referenceType,
                    isFilterByParentId,
                    parentId,
                    isDeleted,
                    sortOption,
                    getPageSize()
                ).subscribeOn(Schedulers.io())
                    .doOnSuccess {
                        token = it.token
                        ids.addAll(it.ids)
                        idPublisher.onNext(ids)
                    }.ignoreElement()
            }
            token!!.isNotEmpty() -> {
                CommentManualLoadWithTokenUsecase().execute(
                    referenceId,
                    referenceType,
                    isFilterByParentId,
                    parentId,
                    isDeleted,
                    token!!
                ).subscribeOn(Schedulers.io())
                    .doOnSuccess {
                        token = it.token
                        ids.addAll(it.ids)
                        idPublisher.onNext(ids)
                    }.ignoreElement()
            }
            else -> {
                // last page reached
                Completable.complete()
            }
        }.doOnTerminate {
            isLoading = false
        }.doOnDispose {
            isLoading = false
        }
    }

    private fun getPageSize(): Int {
        return COMMENT_LOADER_PAGE_SIZE
    }
}