package com.amity.socialcloud.sdk.social.community

import android.os.Parcelable
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.core.events.AmityTopic
import com.amity.socialcloud.sdk.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.core.events.community.AmityCommunityEvents
import com.amity.socialcloud.sdk.core.file.AmityImage
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.amity.socialcloud.sdk.social.feed.AmityFeedType
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.EkoCategoryIds
import com.ekoapp.ekosdk.community.update.AmityCommunityPostSettings
import com.ekoapp.ekosdk.internal.usecase.community.CommunityPostCountUseCase
import com.google.gson.JsonObject
import io.reactivex.Flowable
import kotlinx.android.parcel.Parcelize
import kotlinx.android.parcel.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityCommunity internal constructor(
    private val communityId: String = ObjectId.get().toHexString(),
    private val channelId: String,
    private val userId: String,
    private val displayName: String,
    private val description: String,
    private val avatarFileId: String,
    private val isOfficial: Boolean,
    private val isPublic: Boolean,
    private val onlyAdminCanPost: Boolean,
    private val tags: AmityTags = AmityTags(),
    private val metadata: JsonObject?,
    private val postsCount: Int = 0,
    private val membersCount: Int = 0,
    private val isJoined: Boolean,
    private val isDeleted: Boolean,
    private val categoryIds: EkoCategoryIds,
    internal var categories: List<AmityCommunityCategory>,
    internal var user: AmityUser?,
    internal var avatarImage: AmityImage?,
    private val isPostReviewEnabled: Boolean,
    private val createdAt: DateTime,
    private val updatedAt: DateTime,
    internal val path: String
) : Parcelable {

    fun getCommunityId(): String {
        return communityId
    }

    fun getChannelId(): String {
        return channelId
    }

    fun getUserId(): String {
        return userId
    }

    fun getDisplayName(): String {
        return displayName
    }

    fun getDescription(): String {
        return description
    }

    internal fun getAvatarFileId(): String {
        return avatarFileId
    }

    fun isOfficial(): Boolean {
        return isOfficial
    }

    fun isPublic(): Boolean {
        return isPublic
    }

    @Deprecated("use getPostSettings() instead")
    fun onlyAdminCanPost(): Boolean {
        return onlyAdminCanPost
    }

    fun getTags(): AmityTags {
        return tags
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getPostCount(): Int {
        return postsCount
    }

    fun getMemberCount(): Int {
        return membersCount
    }

    fun isJoined(): Boolean {
        return isJoined
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getUser(): AmityUser? {
        return user
    }

    fun getAvatar(): AmityImage? {
        return avatarImage
    }

    fun getCategoryIds(): EkoCategoryIds {
        return categoryIds
    }

    fun getCategories(): List<AmityCommunityCategory> {
        return categories
    }
    
    @Deprecated("use getPostSettings() instead")
    fun isPostReviewEnabled(): Boolean {
        return isPostReviewEnabled
    }

    fun getPostCount(feedType: AmityFeedType): Flowable<Int> {
        return CommunityPostCountUseCase().execute(communityId, feedType).toV2()
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }
    
    fun getPostSettings(): AmityCommunityPostSettings {
        return when {
            !onlyAdminCanPost && isPostReviewEnabled -> AmityCommunityPostSettings.ADMIN_REVIEW_POST_REQUIRED
            !onlyAdminCanPost && !isPostReviewEnabled -> AmityCommunityPostSettings.ANYONE_CAN_POST
            else -> AmityCommunityPostSettings.ADMIN_CAN_POST_ONLY
        }
    }

    fun subscription(events: AmityCommunityEvents): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.COMMUNITY(this, events))
    }

}