package com.amity.socialcloud.sdk.core.error

import okhttp3.Response
import org.json.JSONObject
import retrofit2.HttpException
import java.io.IOException

open class AmityException protected constructor(
    message: String?,
    cause: Throwable?,
    val code: Int
) : Exception(message, cause) {

    companion object {
        fun create(message: String?, cause: Throwable?, error: AmityError): AmityException {
            return create(message, cause, error.code)
        }

        @JvmStatic
        fun create(message: String?, cause: Throwable?, code: Int): AmityException {
            return AmityException(message, cause, code)
        }

        fun fromThrowable(throwable: Throwable): AmityException {
            // system error
            if (throwable is HttpException) {
                val errorBody = throwable.response()?.errorBody()?.string()
                if (errorBody != null) {
                    return try {
                        val error = JSONObject(errorBody)
                        val code = error.getInt("code")
                        val message = error.getString("message")
                        create(message, null, AmityError.from(code))
                    } catch (e: java.lang.Exception) {
                        create("corrupted error format", null, AmityError.UNKNOWN)
                    }
                }
            }

            // network error
            if (throwable is IOException) {
                return create("network error", null, AmityError.CONNECTION_ERROR)
            }

            // unknown error
            return create("unknown error", null, AmityError.UNKNOWN)
        }

        fun fromResponse(response: Response): AmityException {
            return try {
                val errorBody = response.peekBody(Long.MAX_VALUE).string()
                val error = JSONObject(errorBody)
                val code = error.getInt("code")
                val message = error.getString("message")
                create(message, null, AmityError.from(code))
            } catch (e: java.lang.Exception) {
                create("corrupted error format", null, AmityError.UNKNOWN)
            }
        }
    }
}