package com.ekoapp.ekosdk.internal.api

import android.util.Log
import com.amity.socialcloud.sdk.core.AmityConnectionState
import com.amity.socialcloud.sdk.core.AmityGlobalBanEvent
import com.amity.socialcloud.sdk.core.error.AmityError
import com.amity.socialcloud.sdk.core.error.AmityError.Companion.from
import com.amity.socialcloud.sdk.core.error.AmityException.Companion.create
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.socket.AmitySocketException.Companion.create
import com.amity.socialcloud.sdk.socket.model.SocketRequest
import com.amity.socialcloud.sdk.socket.model.SocketResponse
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.ekosdk.EkoChannelReadStatus
import com.ekoapp.ekosdk.internal.api.EkoEndpoint.getSocketUrl
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.event.*
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelStartReadingsRequest
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.internal.util.RxEko
import com.ekoapp.ekosdk.sdk.BuildConfig
import com.google.common.base.Objects
import com.google.common.collect.ImmutableSet
import com.google.gson.JsonParser
import com.jakewharton.rxrelay2.BehaviorRelay
import hu.akarnokd.rxjava3.bridge.RxJavaBridge
import io.reactivex.BackpressureStrategy
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.PublishSubject
import io.reactivex.subjects.SingleSubject
import io.socket.client.Ack
import io.socket.client.IO
import io.socket.client.Socket
import io.socket.engineio.client.transports.WebSocket
import okhttp3.Dispatcher
import okhttp3.OkHttpClient
import org.json.JSONObject
import timber.log.Timber
import java.net.URISyntaxException
import java.util.*
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicInteger

@Deprecated("")
class EkoSocket(
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
    override fun onSessionStateChange(sessionState: SessionState) {}

    init {
        companionSessionStateEventBus = getSessionStateEventBus()
    }

    override fun establish(account: EkoAccount) {
        terminateSocket()
        currentSocket = init(account)
        currentSocket.connect()
    }
    override fun destroy() {
        terminateSocket()
    }

    override fun handleTokenExpire() {
        terminateSocket()
    }

    fun disconnect() {
        terminateSocket()
    }

    //to properly close a client, followed by socket.io best practice
    //https://socketio.github.io/socket.io-client-java/faq.html#How_to_properly_close_a_client
    private fun terminateSocket() {
        if (currentSocket.connected()) {
            currentSocket.disconnect()
        }
        currentDispatcher.executorService.shutdown()
        // FIXME: remove logs
        val socketHash = Integer.toHexString(currentSocket.hashCode())
        Log.e("socket", "terminate socket: $socketHash" )
    }


    companion object {
        private val TAG = EkoSocket::class.java.name
        private val PROXY = Socket(null, null, null)
        private val rpcId = AtomicInteger(0)
        private var currentAccount = EkoAccount.create("seed")
        private var currentSocket: Socket = PROXY
        private var currentDispatcher: Dispatcher = Dispatcher()
        private val connectionEventRelay = BehaviorRelay.create<SocketConnectionEvent>()
        private val globalBanEventPublisher = PublishSubject.create<AmityGlobalBanEvent>()
        private lateinit var companionSessionStateEventBus: SessionStateEventBus

        @Throws(URISyntaxException::class)
        private fun init(account: EkoAccount): Socket {
            currentDispatcher = Dispatcher()
            currentAccount = account
            val userId = account.userId
            val authority = getSocketUrl()
            Timber.tag(TAG).i("init new socket for: %s , url: %s", userId, authority)
            /*
            Check default value at  io.socket.client.Manager
            reconnectionDelay = 1000
            reconnectionDelayMax = 5000
            randomizationFactor = 0.5
         */

            val okHttpClient: OkHttpClient = OkHttpClient.Builder()
                .dispatcher(currentDispatcher)
                .readTimeout(1, TimeUnit.MINUTES) // important for HTTP long-polling
                .build()
            val options = IO.Options()
            options.callFactory = okHttpClient;
            options.webSocketFactory = okHttpClient;
            options.reconnectionDelayMax = 10000
            options.transports = arrayOf(WebSocket.NAME)
            options.query = String.format("token=%s", account.accessToken)
            val socket = IO.socket(authority, options)
            val events: Set<String> = ImmutableSet.builder<String>()
                .add(Socket.EVENT_CONNECT)
                .add(Socket.EVENT_CONNECT_ERROR)
                .add(Socket.EVENT_CONNECT_TIMEOUT)
                .add(Socket.EVENT_CONNECTING)
                .add(Socket.EVENT_DISCONNECT)
                .add(Socket.EVENT_ERROR)
                .add(Socket.EVENT_RECONNECT)
                .add(Socket.EVENT_RECONNECT_ATTEMPT)
                .add(Socket.EVENT_RECONNECT_FAILED)
                .add(Socket.EVENT_RECONNECTING)
                .add(Socket.EVENT_PING)
                .add(Socket.EVENT_PONG)
                .add(Socket.EVENT_MESSAGE)
                .build()
            for (event in events) {
                socket.on(event) { args: Array<Any?>? ->
                    // FIXME doesn't look good. Find other way?
                    if (Objects.equal(socket, currentSocket)) {
                        val sce = SocketConnectionEvent(userId, socket, event, args!!)
                        this.connectionEventRelay.accept(sce)
                    }
                    if (BuildConfig.DEBUG) {
                        val socketHash = Integer.toHexString(socket.hashCode())
                        Timber.tag(TAG).e(
                            "socket: %s uid: %s connected: %s event: %s args: %s",
                            socketHash,
                            userId,
                            socket.connected(),
                            event,
                            Arrays.deepToString(args)
                        )
                    }
                }
            }
            socket.on(Socket.EVENT_DISCONNECT) { args: Array<Any?> ->
                if (args.size > 0 && Objects.equal(
                        args[0], "io server disconnect"
                    )
                ) {
                    socket.connect()
                }
            }
            socket.on(Socket.EVENT_ERROR) { args: Array<Any?>? ->
                try {
                    val parser = JsonParser()
                    val element = parser.parse(Arrays.deepToString(args))
                    val array = element.asJsonArray
                    val `object` = array[0].asJsonObject
                    val exception = create(
                        `object`["message"].asString,
                        null,
                        `object`["code"].asInt
                    )
                    if (from(exception).`is`(AmityError.USER_IS_GLOBAL_BANNED)) {
                        globalBanEventPublisher.onNext(AmityGlobalBanEvent(userId))
                        companionSessionStateEventBus.publish(SessionState.Terminated(exception))
                    }
                } catch (e: Exception) {
                    Timber.tag(TAG)
//                        .e(e, String.format("event: error arg: %s", Arrays.deepToString(args)))
                }
            }
            Completable.fromAction {
                val channelDao = UserDatabase.get().channelDao()
                channelDao.deleteAllLocallyInactiveChannelsAndUpdateAllActiveChannelsToNotReading()
            }
                .doOnComplete {
                    socket.on(Socket.EVENT_CONNECT) { args: Array<Any?>? ->
                        callStartReadingOnAllChannelsWithReadingStatus()
                            .subscribe()
                    }
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
            subscribeSocketEvent(socket, ChannelDidBanListener())
            subscribeSocketEvent(socket, ChannelDidCreateListener())
            subscribeSocketEvent(socket, ChannelDidJoinListener())
            subscribeSocketEvent(socket, ChannelDidLeaveListener())
            subscribeSocketEvent(socket, ChannelDidMarkSeenListener())
            subscribeSocketEvent(socket, ChannelDidUnbanListener())
            subscribeSocketEvent(socket, ChannelDidUpdateListener())
            subscribeSocketEvent(socket, ChannelDidAddUsersListener())
            subscribeSocketEvent(socket, ChannelDidRemoveUsersListener())
            subscribeSocketEvent(socket, ChannelDidDeleteListener())
            subscribeSocketEvent(socket, MessageDidCreateListener())
            subscribeSocketEvent(socket, MessageDidDeleteListener())
            subscribeSocketEvent(socket, MessageDidUpdateListener())
            subscribeSocketEvent(socket, UserDidUpdateListener())
            subscribeSocketEvent(socket, StreamDidStartListener())
            subscribeSocketEvent(socket, StreamDidStopListener())
            return socket
        }

        private fun callStartReadingOnAllChannelsWithReadingStatus(): Completable {
            val extraDao = UserDatabase.get().channelExtraDao()
            return RxJavaBridge.toV2Single(extraDao.getAllIdsByReadStatus(EkoChannelReadStatus.READING))
                .filter { channelIds: List<String?> -> channelIds.isNotEmpty() }
                .flatMapCompletable { channelIds: List<String>? ->
                    rpc(ChannelStartReadingsRequest(channelIds))
                        .ignoreElement()
                }
        }

        private fun subscribeSocketEvent(socket: Socket, listener: SocketEventListener) {
            socket.on(listener.event, listener)
        }

        @JvmStatic
        fun connectionEvent(): Flowable<SocketConnectionEvent> {
            return this.connectionEventRelay.toFlowable(BackpressureStrategy.BUFFER)
        }

        val connectionState: Flowable<AmityConnectionState>
            get() = connectionEvent()
                .map { socketConnectionEvent: SocketConnectionEvent ->
                    when (socketConnectionEvent.event) {
                        Socket.EVENT_CONNECT -> return@map AmityConnectionState.CONNECTED
                        Socket.EVENT_CONNECTING -> return@map AmityConnectionState.CONNECTING
                        Socket.EVENT_DISCONNECT -> return@map AmityConnectionState.DISCONNECTED
                        Socket.EVENT_CONNECT_ERROR, Socket.EVENT_RECONNECT_FAILED, Socket.EVENT_ERROR -> return@map AmityConnectionState.FAILED
                        Socket.EVENT_RECONNECTING -> return@map AmityConnectionState.RECONNECTING
                        Socket.EVENT_PONG -> if (socketConnectionEvent.isConnected) {
                            return@map AmityConnectionState.CONNECTED
                        } else {
                            return@map AmityConnectionState.DISCONNECTED
                        }
                    }
                    AmityConnectionState.UNKNOWN
                }
                .filter { amityConnectionState: AmityConnectionState -> amityConnectionState !== AmityConnectionState.UNKNOWN }
                .distinctUntilChanged()

        @JvmStatic
        fun <T> call(call: Call<T>): Single<T> {
            val subject = SingleSubject.create<T>()
            rpc(call.request)
                .map { response: SocketResponse? -> call.converter.convert(response) }
                .subscribe(subject)
            return subject.hide()
        }

        val globalBanEvents: Flowable<AmityGlobalBanEvent>
            get() = globalBanEventPublisher.toFlowable(BackpressureStrategy.BUFFER)

        fun rpc(request: SocketRequest): Single<SocketResponse> {
            return rpc(request.method(), request)
        }

        @JvmOverloads
        fun rpc(method: String, parameter: Any? = null): Single<SocketResponse> {
            val responseSubject = SingleSubject.create<SocketResponse>()
            val json = EkoGson.get().toJson(parameter)
            val orgJson = JSONObject(json)
            wrapConnectionError(currentSocket, method, responseSubject)
            if (currentSocket.connected()) {
                emit(currentSocket, method, orgJson, responseSubject)
            } else {
                currentSocket.once(Socket.EVENT_CONNECT) { args: Array<Any?>? ->
                    emit(
                        currentSocket,
                        method,
                        orgJson,
                        responseSubject
                    )
                }
                Timber.tag(TAG).w("rpc: reschedule: %s parameter: %s", method, orgJson)
            }
            return responseSubject.doOnError(RxEko.CATCH_UNAUTHORIZED_ERROR_CONSUMER).hide()
        }

        private fun wrapConnectionError(
            socketio: Socket,
            method: String,
            responseSubject: SingleSubject<SocketResponse>
        ) {
            val errors = arrayOf(Socket.EVENT_CONNECT_ERROR, Socket.EVENT_ERROR)
            for (errorEvent in errors) {
                socketio.once(errorEvent) { args: Array<Any?>? ->
                    val msg = errorEvent + ": " + Arrays.deepToString(args)
                    val code = AmityError.CONNECTION_ERROR.code
                    val amityError = create(msg, code)
                    if (code == AmityError.USER_IS_GLOBAL_BANNED.code) {
                        companionSessionStateEventBus.publish(SessionState.Terminated(amityError))
                    }
                    responseSubject.onError(amityError)
                    Timber.tag(TAG).e("rpc: %s error (%s): %s", method, code, msg)
                }
            }
        }

        private fun emit(
            socket: Socket,
            method: String,
            parameter: Any?,
            responseSubject: SingleSubject<SocketResponse>
        ) {
            val rpcId = rpcId.getAndIncrement()
            Timber.tag(TAG).i("rpc: %s [%s] parameter: %s", method, rpcId, parameter)
            socket.emit(method, parameter, Ack { args ->
                val response = EkoGson.get()
                    .fromJson(args[0].toString(), SocketResponse::class.java)
                if (response.isSuccess) {
                    Timber.tag(TAG).i(
                        "rpc: %s [%s] success socket response: %s",
                        method,
                        rpcId,
                        Arrays.deepToString(args)
                    )
                    responseSubject.onSuccess(response)
                } else {
                    val msg = response.message
                    val code = response.code
                    Timber.tag(TAG).e(
                        "rpc: %s [%s] error socket response (%s): %s",
                        method,
                        rpcId,
                        code,
                        Arrays.deepToString(args)
                    )
                    responseSubject.onError(create(msg, code))
                }
            })
        }
    }
}