package com.ekoapp.ekosdk.internal.api.event;

import androidx.annotation.NonNull;

import com.amity.socialcloud.sdk.chat.data.message.MessageEventPersister;
import com.amity.socialcloud.sdk.socket.util.EkoGson;
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto;
import com.ekoapp.ekosdk.internal.api.dto.EkoMessageDto;
import com.ekoapp.ekosdk.internal.api.dto.EkoUserDto;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.dao.EkoChannelDao;
import com.ekoapp.ekosdk.internal.data.dao.EkoMessageDao;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.jakewharton.rxrelay2.PublishRelay;
import com.jakewharton.rxrelay2.Relay;

import org.joda.time.DateTime;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import hu.akarnokd.rxjava3.bridge.RxJavaBridge;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Predicate;
import io.reactivex.schedulers.Schedulers;

abstract class MessageListener extends SocketEventListener {

    private static final long BUFFER_TIME_SPAN_IN_MILLIS = 500;
    private static final int BUFFER_MAX_COUNT = 100;
    private static final int RETAIN_MESSAGES_PER_CHANNEL = 1000;

    private final Gson gson = EkoGson.get();
    private final EkoMessageDao messageDao = UserDatabase.get().messageDao();
    private final Relay<MessageQueryDto> relay = PublishRelay.create();

    private final EkoChannelDao channelDao = UserDatabase.get().channelDao();


    MessageListener() {
        relay.buffer(BUFFER_TIME_SPAN_IN_MILLIS, TimeUnit.MILLISECONDS, Schedulers.io(), BUFFER_MAX_COUNT)
                .filter(hasItem)
                .doOnNext(persist)
                .doOnNext(trim)
                .subscribe();
    }

    @Override
    protected void onEvent(@NonNull String event, Object... args) {
        super.onEvent(event, args);
        String json = args[0].toString();
        MessageQueryDto dto = gson.fromJson(json, MessageQueryDto.class);
        relay.accept(dto);
        onMessageEvent(dto.getMessages());
    }

    private void onMessageEvent(@NonNull List<EkoMessageDto> messages) {
        Map<String, String> channelIds = Maps.newConcurrentMap();
        for (EkoMessageDto message : messages) {
            String channelId = message.getChannelId();
            if(channelId != null) {
                channelIds.put(channelId, channelId);
            }
        }
        for (String channelId : channelIds.values()) {
            channelDao.updateLastActivity(channelId, DateTime.now());
        }
    }

    private Predicate<List<MessageQueryDto>> hasItem = list -> !list.isEmpty();

    private Consumer<List<MessageQueryDto>> persist = list -> {
        MessageQueryDto dto = list.get(0);
        final int size = list.size();
        if (size > 1) {
            List<EkoMessageDto> messages = dto.getMessages();
            List<EkoUserDto> users = dto.getUsers();
            for (int i = 1; i < size; i++) {
                MessageQueryDto item = list.get(i);
                messages.addAll(item.getMessages());
                users.addAll(item.getUsers());
            }
        }
        RxJavaBridge.toV2Completable(new MessageEventPersister().persist(dto))
                .subscribeOn(Schedulers.io())
                .doOnError(new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) throws Exception {
                        // do nothing
                    }
                })
                .subscribe();
    };

    private Consumer<List<MessageQueryDto>> trim = list -> {
        EkoMessageDto dto = list.get(0).getMessages().get(0);
        final String channelId = dto.getChannelId();
        messageDao.retainLatestFromChannel(channelId, RETAIN_MESSAGES_PER_CHANNEL);
    };
}
