package com.ekoapp.ekosdk.internal.api.socket.call;

import com.amity.socialcloud.sdk.core.reaction.ReactionReferenceType;
import com.amity.socialcloud.sdk.socket.model.SocketResponse;
import com.ekoapp.ekosdk.internal.api.dto.EkoReactionAndUserListDto;
import com.ekoapp.ekosdk.internal.api.dto.EkoReactionQueryResultDto;
import com.ekoapp.ekosdk.internal.api.mapper.EkoReactionMapper;
import com.ekoapp.ekosdk.internal.api.mapper.EkoObjectMapper;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.dao.EkoReactionDao;
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity;
import com.google.common.collect.Maps;

import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class ReactionQueryConverter implements ResponseConverter<EkoReactionQueryResultDto> {

    final private EkoReactionDao reactionDao;

    final String referenceId;
    final String referenceType;
    final String lastMapReactionId;

    public ReactionQueryConverter(String referenceId, String referenceType, String lastMapReactionId) {
        this.reactionDao = UserDatabase.get().reactionDao();
        this.referenceId = referenceId;
        this.referenceType = referenceType;
        this.lastMapReactionId = lastMapReactionId;
    }

    public EkoObjectMapper<EkoReactionAndUserListDto, List<EkoReactionEntity>> getMapper() {
        return EkoReactionMapper.MAPPER;
    }

    public enum PERSIST_PAGE_TYPE {

        NO_DATA(false, false, false),
        ONE_PAGE(false, false, true),
        FIRST_PAGE(true, false, true),
        REGULAR_PAGE(true, true, true),
        LAST_PAGE(false, true, true),

        // Remove this case, when backend issue is fixed.
        // Known issue: When result item count is the same as request page size,
        // Backend returns next token, where a query with that token returns an empty last page.
        EMPTY_LAST_PAGE(false, true, false);

        boolean hasNextToken;
        boolean hasPrevToken;
        boolean hasItem;

        PERSIST_PAGE_TYPE(boolean hasNextToken, boolean hasPrevToken, boolean hasItem) {
            this.hasNextToken = hasNextToken;
            this.hasPrevToken = hasPrevToken;
            this.hasItem = hasItem;
        }

    }

    private PERSIST_PAGE_TYPE getPersistPageType(boolean hasNextToken, boolean hasPrevToken, boolean hasItem) {
        PERSIST_PAGE_TYPE pageType = PERSIST_PAGE_TYPE.NO_DATA;
        for (PERSIST_PAGE_TYPE value : PERSIST_PAGE_TYPE.values()) {
            if (value.hasNextToken == hasNextToken
                    && value.hasPrevToken == hasPrevToken
                    && value.hasItem == hasItem) {
                pageType = value;
                break;
            }
        }
        return pageType;
    }

    @Override
    public EkoReactionQueryResultDto convert(SocketResponse response) {
        EkoReactionQueryResultDto dto = response.getData(EkoReactionQueryResultDto.class);
        String next = dto.getToken().getNext();
        String previous = dto.getToken().getPrevious();
        List<EkoReactionEntity> remoteReactionList = getMapper().map(dto.getResults());

        PERSIST_PAGE_TYPE pageType = getPersistPageType(!StringUtils.isEmpty(next),
                !StringUtils.isEmpty(previous),
                (remoteReactionList != null && !remoteReactionList.isEmpty()));

        if (pageType == PERSIST_PAGE_TYPE.NO_DATA) {
            removeAllReactions(referenceId, referenceType);
        } else {
            persistReactionByPageType(pageType, remoteReactionList);
        }

        return dto;
    }

    private void persistReactionByPageType(PERSIST_PAGE_TYPE pageType, List<EkoReactionEntity> remoteList) {
        List<EkoReactionEntity> onDiskList = new ArrayList<>();
        switch (pageType) {

            case NO_DATA:
            case ONE_PAGE: {
                onDiskList = reactionDao.getAllByReferenceTypeAndReferenceId(referenceType, referenceId);
                break;
            }

            case FIRST_PAGE: {
                final String lastReactionIdFromRemote = remoteList.get(remoteList.size() - 1).getReactionId();
                onDiskList = reactionDao.getAllAfterReactionId(lastReactionIdFromRemote);
                break;
            }

            case REGULAR_PAGE: {
                final String lastReactionIdFromRemote = remoteList.get(remoteList.size() - 1).getReactionId();
                onDiskList = reactionDao.getAllBetweenReactionId(lastReactionIdFromRemote, lastMapReactionId);
                break;
            }

            case EMPTY_LAST_PAGE:
            case LAST_PAGE: {
                onDiskList = reactionDao.getAllBeforeReactionId(lastMapReactionId);
                break;
            }

            default:
                break;
        }

        persistReaction(onDiskList, remoteList);
    }

    private void persistReaction(List<EkoReactionEntity> onDiskList, List<EkoReactionEntity> remoteList) {
        Map<String, EkoReactionEntity> remoteMap = Maps.newHashMap();
        for (EkoReactionEntity reaction : remoteList) {
            EkoReactionEntity internalReaction = EkoReactionEntity.create(reaction.getReactionId(),
                    ReactionReferenceType.Companion.enumOf(referenceType),
                    reaction.getReferenceId(),
                    reaction.getReactionName(),
                    reaction.getUserId(),
                    reaction.getUserDisplayName(),
                    reaction.getCreatedAt(),
                    reaction.getUpdatedAt()
                    );
            remoteMap.put(reaction.getId(), internalReaction);
        }

        List<EkoReactionEntity> updateList = new ArrayList<>();
        List<EkoReactionEntity> deleteList = new ArrayList<>();

        for (EkoReactionEntity onDiskReaction : onDiskList) {
            String onDiskId = onDiskReaction.getId();
            if (remoteMap.containsKey(onDiskId)) {
                updateList.add(remoteMap.get(onDiskId));
                remoteMap.remove(onDiskId);
            } else {
                deleteList.add(onDiskReaction);
            }
        }

        List<EkoReactionEntity> insertList = new ArrayList<>(remoteMap.values());

        UserDatabase.get().runInTransaction(() -> {
            deleteReactions(deleteList);
            updateReactions(updateList);
            insertReactions(insertList);
        });
    }

    private void insertReactions(List<EkoReactionEntity> reactionList) {
        reactionDao.insert(reactionList);
    }

    private void updateReactions(List<EkoReactionEntity> reactionList) {
        reactionDao.update(reactionList);
    }

    private void deleteReactions(List<EkoReactionEntity> reactionList) {
        reactionDao.delete(reactionList);
    }

    private void removeAllReactions(String referenceId, String referenceType) {
        reactionDao.deleteByReferenceId(referenceId, referenceType);
    }

}