package com.ekoapp.ekosdk.internal.data.boundarycallback;

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import androidx.paging.PagedList.BoundaryCallback
import com.amity.socialcloud.sdk.core.reaction.AmityReaction
import com.amity.socialcloud.sdk.core.reaction.ReactionReferenceType
import com.ekoapp.core.utils.getCurrentClassAndMethodNames
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.dto.EkoReactionAndUserListDto
import com.ekoapp.ekosdk.internal.api.dto.EkoReactionQueryResultDto
import com.ekoapp.ekosdk.internal.api.mapper.EkoObjectMapper
import com.ekoapp.ekosdk.internal.api.mapper.EkoReactionMapper
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.ReactionQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.call.ResponseConverter
import com.ekoapp.ekosdk.internal.api.socket.request.ReactionQueryRequest
import com.ekoapp.ekosdk.internal.api.socket.request.ReactionQueryRequest.ReactionQueryOptions
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import com.google.common.collect.Maps
import com.google.common.collect.Sets
import io.reactivex.Completable
import io.reactivex.CompletableObserver
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class AmityReactionBoundaryCallback(
    val referenceId: String,
    val referenceType: ReactionReferenceType,
    val pageSize: Int,
    val delaySubject: Subject<Boolean>
) : BoundaryCallback<AmityReaction>(), CompletableObserver, Function<AmityReaction, AmityReaction> {

    private val mapReactionIdAndToken: MutableMap<String, Pair<String?, Boolean>> =
        Maps.newConcurrentMap()
    private val setOfReactionId = Sets.newConcurrentHashSet<String>()

    val mapper: EkoObjectMapper<EkoReactionAndUserListDto, List<EkoReactionEntity>>
        get() = EkoReactionMapper.MAPPER

    fun getConverter(
        referenceId: String,
        referenceType: String,
        lastMapReactionId: String?
    ): ResponseConverter<EkoReactionQueryResultDto> {
        return ReactionQueryConverter(referenceId, referenceType, lastMapReactionId)
    }

    override fun onItemAtFrontLoaded(itemAtFront: AmityReaction) {
        // TODO: 2019-08-31
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityReaction) {
        // TODO: 2019-08-31
    }

    override fun onZeroItemsLoaded() {
        // TODO: 2019-08-31
    }

    fun onFirstLoaded() {
        val TAG = getCurrentClassAndMethodNames()
        AmityLog.tag(TAG).i("onFirstLoaded")
        val options = ReactionQueryOptions()
        options.limit = pageSize
        call(options)
            .doOnComplete { delaySubject.onComplete() }
            .doOnError { delaySubject.onComplete() }
            .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
            .subscribe(this)
    }

    override fun apply(input: AmityReaction): AmityReaction {
        setOfReactionId.add(input.getReactionId())
        mapByReaction(input.getReactionId())
        return input
    }

    private fun mapByReaction(reactionId: String) {
        mapReactionIdAndToken[reactionId]?.let { tokenAndStatusNonNull ->
            if (tokenAndStatusNonNull.first.isNullOrEmpty() || tokenAndStatusNonNull.second == true) {
                return
            }
            val TAG = getCurrentClassAndMethodNames()
            AmityLog.tag(TAG).i("map reactionId:$reactionId")
            val options = ReactionQueryOptions()
            options.token = tokenAndStatusNonNull.first
            call(options, reactionId)
                .doOnSubscribe {
                    mapReactionIdAndToken[reactionId] = Pair(tokenAndStatusNonNull.first, true)
                }
                .doOnError {
                    mapReactionIdAndToken[reactionId] = Pair(tokenAndStatusNonNull.first, false)
                }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
        }
    }

    private fun call(
        options: ReactionQueryOptions,
        lastMapReactionId: String? = null
    ): Completable {
        val reactionQueryRequest = ReactionQueryRequest()
        reactionQueryRequest.referenceId = referenceId
        reactionQueryRequest.referenceType = referenceType.value
        reactionQueryRequest.options = options
        return EkoSocket.call(
            Call.create(
                reactionQueryRequest,
                getConverter(referenceId, referenceType.value, lastMapReactionId)
            )
        )
            .doOnSuccess { dto: EkoReactionQueryResultDto ->
                val reactionList = mapper.map(dto.results)
                if (reactionList.isNotEmpty()) {
                    val reaction = reactionList[reactionList.size - 1]
                    val lastObtainReactionId = reaction.getReactionId()
                    mapReactionIdAndToken[lastObtainReactionId] = Pair.create(dto.token.next ?: "", false)
                    if (setOfReactionId.contains(lastObtainReactionId)) {
                        mapByReaction(lastObtainReactionId)
                    }
                }
            }
            .retryWhen(RetryWhen.maxRetries(3)
                .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                .action { errorAndDuration: ErrorAndDuration ->
                    AmityLog.tag(
                        getCurrentClassAndMethodNames()
                    ).e(
                        errorAndDuration.throwable(),
                        "an error occurred, back-off for durationMs:%s",
                        errorAndDuration.durationMs()
                    )
                }
                .build())
            .ignoreElement()
    }

    override fun onSubscribe(d: Disposable) {}
    override fun onComplete() {}
    override fun onError(e: Throwable) {
        val TAG = getCurrentClassAndMethodNames()
        AmityLog.tag(TAG).e(e)
    }

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }

    init {
        onFirstLoaded()
    }
}