package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.social.feed.AmityPost
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.PostQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityFeedQueryRequest
import com.ekoapp.ekosdk.internal.api.socket.request.FeedQueryOptions
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.CommunityFeedQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.CommunityFeedQueryTokenEntity
import io.reactivex.Completable
import io.reactivex.schedulers.Schedulers

// unix page number for backward compat!
private const val PAGE_NUMBER = 1

internal class CommunityFeedBoundaryCallback(
        private val communityId: String,
        private val sortOption: String,
        private val isDeleted: Boolean?,
        private val feedType: String,
        val pageSize: Int
) : PagedList.BoundaryCallback<AmityPost>() {

    init {
        onFirstLoaded()
    }

    private val tokenDao: CommunityFeedQueryTokenDao = UserDatabase.get().communityFeedQueryTokenDao()

    private fun onFirstLoaded() {
        val request = CommunityFeedQueryRequest(
                targetType = "community",
                targetId = communityId,
                feedType = feedType,
                sortBy = sortOption,
                isDeleted = isDeleted,
                options = FeedQueryOptions(limit = pageSize)
        )
        call(request)
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityPost) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "communityId" to communityId,
                        "sortBy" to sortOption,
                        "isDeleted" to (isDeleted?.toString() ?: ""),
                        "feedType" to feedType
                ), pageNumber = PAGE_NUMBER
        )
                .filter { token: CommunityFeedQueryTokenEntity -> !token.next.isNullOrEmpty() }
                .flatMapCompletable { token: CommunityFeedQueryTokenEntity ->
                    call(
                            CommunityFeedQueryRequest(
                                    targetType = "community",
                                    targetId = communityId,
                                    sortBy = sortOption,
                                    isDeleted = isDeleted,
                                    feedType = feedType,
                                    options = FeedQueryOptions(token = token.next!!)
                            )
                    )
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun call(request: CommunityFeedQueryRequest): Completable {
        return EkoSocket.call(Call.create(request, PostQueryConverter()))
                .doOnSuccess { dto ->
                    val token = CommunityFeedQueryTokenEntity().apply {
                        this.communityId = this@CommunityFeedBoundaryCallback.communityId
                        this.isDeleted = this@CommunityFeedBoundaryCallback.isDeleted?.toString() ?: ""
                        this.sortBy = this@CommunityFeedBoundaryCallback.sortOption
                        this.feedType = this@CommunityFeedBoundaryCallback.feedType
                        this.pageNumber = PAGE_NUMBER
                        this.next = dto.paging?.next
                        this.previous = dto.paging.previous
                        this.ids = dto.posts?.map { it.postId }
                        this.updatedAt = AmityCoreClient.getServerTime()
                    }
                    val isFirstPage = dto.paging?.previous == null
                    if (isFirstPage) {
                        tokenDao.insertToken(token)
                                .subscribeOn(Schedulers.io())
                                .subscribe()
                    } else {
                        updateQueryToken(token)
                    }
                }
                .ignoreElement()
    }

    private fun updateQueryToken(token: CommunityFeedQueryTokenEntity) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "communityId" to communityId,
                        "sortBy" to sortOption,
                        "isDeleted" to (isDeleted?.toString() ?: ""),
                        "feedType" to feedType
                ), pageNumber = PAGE_NUMBER
        ).flatMapCompletable {
            token.ids.addAll(0, it.ids)
            tokenDao.insertToken(token)
        }.subscribeOn(Schedulers.io())
                .subscribe()
    }

}