package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import com.amity.socialcloud.sdk.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.chat.channel.AmityChannelFilter
import com.amity.socialcloud.sdk.chat.data.channel.ChannelQueryPersister
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRemoteDataStore
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.ekoapp.core.utils.getCurrentClassAndMethodNames
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.ChannelQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelQueryRequest.ChannelQueryOptions
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import com.google.common.collect.Maps
import com.google.common.collect.Sets
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class EkoChannelBoundaryCallback(
    private val channelTypes: Array<String>,
    private val filter: AmityChannelFilter,
    private val isDeleted: Boolean?,
    includingTags: AmityTags,
    excludingTags: AmityTags,
    pageSize: Int,
    private val delaySubject: Subject<Boolean>
) : EkoWithTagBoundaryCallback<AmityChannel>(includingTags, excludingTags, pageSize),
    Function<AmityChannel, AmityChannel> {

    init {
        onFirstLoaded()
    }

    private val channelIdAndTokenMap: MutableMap<String, Pair<String, Boolean>> =
        Maps.newConcurrentMap()
    private val channelIdSet = Sets.newConcurrentHashSet<String>()

    override fun onItemAtFrontLoaded(itemAtFront: AmityChannel) {
        // TODO: 2019-08-31
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityChannel) {
        // TODO: 2019-08-31
    }

    override fun onZeroItemsLoaded() {
        // TODO: 2019-08-31
    }

    override fun onFirstLoaded() {
        val TAG = getCurrentClassAndMethodNames()
        AmityLog.tag(TAG).i("onFirstLoaded")
        val options = ChannelQueryOptions(limit = pageSize)
        call(options)
            .doOnComplete { delaySubject.onComplete() }
            .doOnError { delaySubject.onComplete() }
            .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
            .subscribe(this)
    }

    override fun apply(input: AmityChannel): AmityChannel {
        channelIdSet.add(input.getChannelId())
        mapByChannel(input.getChannelId())
        return input
    }

    private fun mapByChannel(channelId: String) {
        channelIdAndTokenMap[channelId]?.let { mapChannelIdAndTokenNonNull ->
            if (mapChannelIdAndTokenNonNull.first.isNullOrEmpty() || mapChannelIdAndTokenNonNull.second == true) {
                return
            }
            val TAG = getCurrentClassAndMethodNames()
            AmityLog.tag(TAG).i("map channelId:$channelId")
            val options = ChannelQueryOptions(token = mapChannelIdAndTokenNonNull.first)
            call(options)
                .doOnSubscribe {
                    channelIdAndTokenMap[channelId] = Pair(mapChannelIdAndTokenNonNull.first, true)
                }
                .doOnError {
                    channelIdAndTokenMap[channelId] = Pair(mapChannelIdAndTokenNonNull.first, false)
                }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
        }
    }

    private fun call(options: ChannelQueryOptions): Completable {
        return ChannelRemoteDataStore().queryChannels(
            types = listOf(*channelTypes),
            filter = filter.apiKey,
            includingTags = includingTags,
            excludingTags = excludingTags,
            options = QueryOptionsRequestParams(options.limit, options.token)
        ).toV2()
            .flatMap {
                ChannelQueryPersister().persist(it).toV2()
                    .andThen(Single.just(it))
            }
            .doOnSuccess { dto: ChannelQueryDto ->
                val channels = dto.channelDtoList
                if (!channels.isNullOrEmpty()) {
                    val channelId =
                        channels.lastOrNull { if (this.isDeleted == null) true else this.isDeleted == it.deleted }?.channelId
                    val nextPage = dto.token.next
                    if (!nextPage.isNullOrEmpty()) {
                        if (!channelId.isNullOrEmpty()) {
                            channelIdAndTokenMap[channelId] = Pair.create(nextPage, false)
                            if (channelIdSet.contains(channelId)) {
                                mapByChannel(channelId)
                            }
                        } else {
                            val options = ChannelQueryOptions(token = nextPage)
                            call(options)
                                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                                .subscribe(this)
                        }
                    }
                }
            }
            .retryWhen(RetryWhen.maxRetries(3)
                .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                .action { errorAndDuration: ErrorAndDuration ->
                    val TAG = getCurrentClassAndMethodNames()
                    AmityLog.tag(TAG).e(
                        errorAndDuration.throwable(),
                        "an error occurred, back-off for durationMs:%s",
                        errorAndDuration.durationMs()
                    )
                }
                .build())
            .ignoreElement()
    }

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }

}