package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList.BoundaryCallback
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMember
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembership
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembershipFilter
import com.amity.socialcloud.sdk.chat.channel.AmityChannelMembershipSortOption
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipQueryPersister
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRemoteDataStore
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.core.user.AmityMembershipType
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.ChannelMembershipQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelMembershipQueryRequest
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelMembershipQueryRequest.ChannelMembershipQueryOptions
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.converter.EkoRolesTypeConverter
import com.ekoapp.ekosdk.internal.data.dao.EkoChannelMembershipQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.EkoChannelMembershipQueryToken
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers

// unix page number for backward compat!
private const val PAGE_NUMBER = -1

internal class EkoChannelMembershipBoundaryCallback(
    private val channelId: String,
    private val memberships: List<AmityMembershipType>,
    private val isMuted: Boolean?,
    private val roles: AmityRoles,
    private val sortOption: AmityChannelMembershipSortOption,
    private val pageSize: Int
) : BoundaryCallback<AmityChannelMember>() {

    private val tokenDao: EkoChannelMembershipQueryTokenDao =
        UserDatabase.get().channelMembershipQueryTokenDao()

    private fun onFirstLoaded() {
        Completable.fromAction { deleteMembershipByChannelId() }
            .andThen(
                call(
                    QueryOptionsRequestParams(
                        token = null,
                        limit = pageSize
                    )
                )
            )
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    private fun deleteMembershipByChannelId() {
        UserDatabase.get()
            .channelMembershipDao()
            .deleteAllFromChannel(channelId)
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityChannelMember) {
        tokenDao.getQueryTokenByPageNumber(
            primaryKeys = mapOf(
                "channelId" to channelId,
                "roles" to roles,
                "filter" to getFilter(memberships, isMuted),
                "sortBy" to sortOption.apiKey
            ), pageNumber = PAGE_NUMBER
        )
            .filter { token: EkoChannelMembershipQueryToken -> !token.next.isNullOrEmpty() }
            .flatMapCompletable { token: EkoChannelMembershipQueryToken ->
                call(
                    QueryOptionsRequestParams(
                        token = token.next,
                        limit = null
                    )
                )
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    private fun call(options: QueryOptionsRequestParams): Completable {
        return ChannelMembershipRemoteDataStore().queryChannelMembers(
            channelId = channelId,
            roles = roles.takeIf { it.isNotEmpty() },
            memberships = memberships.map { it.apiKey }.plus( if(isMuted != null) listOf(AmityChannelMembership.MUTED.apiKey) else emptyList()),
            sortBy = sortOption.apiKey,
            options = options
        ).toV2()
            .flatMap {
                ChannelMembershipQueryPersister().persist(it).toV2()
                    .andThen(Single.just(it))
            }
            .doOnSuccess { dto: ChannelMembershipQueryDto ->
                val token = dto.token
                token?.apply {
                    channelId = this@EkoChannelMembershipBoundaryCallback.channelId
                    roles = this@EkoChannelMembershipBoundaryCallback.roles
                    filter = getFilter(memberships, isMuted)
                    sortBy = sortOption.apiKey
                    pageNumber = PAGE_NUMBER

                }?.run {
                    tokenDao.insertToken(this)
                }
            }
            .ignoreElement()
    }

    private fun getFilter(
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?
    ): AmityChannelMembershipFilter {
        return when (isMuted) {
            null -> {
                if(memberships.contains(AmityMembershipType.MEMBER)
                    && memberships.contains(AmityMembershipType.BANNED)){
                    AmityChannelMembershipFilter.ALL
                } else if (memberships.contains(AmityMembershipType.MEMBER)){
                    AmityChannelMembershipFilter.MEMBER
                } else if (memberships.contains(AmityMembershipType.BANNED)){
                    AmityChannelMembershipFilter.BANNED
                } else {
                    AmityChannelMembershipFilter.ALL
                }
            }
            true -> {
                AmityChannelMembershipFilter.MUTED
            }
            else -> {
                AmityChannelMembershipFilter.ALL
            }
        }
    }

    init {
        onFirstLoaded()
    }

}