package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import com.amity.socialcloud.sdk.social.comment.AmityComment
import com.amity.socialcloud.sdk.social.comment.AmityCommentSortOption
import com.amity.socialcloud.sdk.social.data.comment.CommentLocalDataStore
import com.amity.socialcloud.sdk.social.data.comment.CommentQueryPersister
import com.amity.socialcloud.sdk.social.data.comment.CommentRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.EkoCommentAndUserListDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommentQueryRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CommentQueryRequest.CommentQueryRequestOptions
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import com.google.common.collect.Maps
import com.google.common.collect.Sets
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class EkoCommentBoundaryCallback(
    private val referenceId: String,
    private val referenceType: String,
    private val isFilterByParentId: Boolean,
    private val parentId: String?,
    private val isDeleted: Boolean?,
    private val sortBy: AmityCommentSortOption,
    pageSize: Int,
    private val delaySubject: Subject<Boolean>
) : EkoBoundaryCallback<AmityComment>(pageSize), Function<AmityComment, AmityComment> {

    private val commentIdAndTokenMap: MutableMap<String, Pair<String, Boolean>> =
        Maps.newConcurrentMap()
    private val commentIdSet = Sets.newConcurrentHashSet<String>()

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }

    init {
        onFirstLoaded()
    }

    override fun onItemAtFrontLoaded(itemAtFront: AmityComment) {
        // TODO: 2019-08-31
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityComment) {
        // TODO: 2019-08-31
    }

    override fun onZeroItemsLoaded() {
        // TODO: 2019-08-31
    }

    override fun onFirstLoaded() {
        val options = CommentQueryRequestOptions()
        options.limit = pageSize
        options.type = "pagination"

        AmityLog.tag(EkoCommentBoundaryCallback::class.java.name).i("onFirstLoaded")
        call(options)
            .doOnComplete {
                delaySubject.onComplete()
            }
            .doOnError {
                delaySubject.onComplete()
            }
            .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
            .subscribe(this)
    }

    override fun apply(input: AmityComment): AmityComment {
        commentIdSet.add(input.getCommentId())
        mapByComment(input.getCommentId())
        return input
    }

    private fun mapByComment(commentId: String) {
        commentIdAndTokenMap[commentId]?.let { tokenAndStatusNonNull ->
            if (tokenAndStatusNonNull.first.isNullOrEmpty() || tokenAndStatusNonNull.second == true) {
                return
            }
            val options = CommentQueryRequestOptions(token = tokenAndStatusNonNull.first)
            options.type = "pagination"

            call(options)
                .doOnSubscribe {
                    commentIdAndTokenMap[commentId] = Pair(tokenAndStatusNonNull.first, true)
                }
                .doOnError {
                    commentIdAndTokenMap[commentId] = Pair(tokenAndStatusNonNull.first, false)
                }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
        }
    }

    @Deprecated("hack before moving to paging data")
    private fun clearComments(): Completable {
        val commentLocalDataStore = CommentLocalDataStore()
        return commentLocalDataStore.deleteByReferenceId(referenceId)
            .toV2()
            .subscribeOn(Schedulers.io())
    }

    private fun call(options: CommentQueryRequestOptions): Completable {
        val commentQueryRequest = CommentQueryRequest(
            referenceId = referenceId,
            referenceType = referenceType,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            sortBy = sortBy.apiKey,
            options = options
        )

        return CommentRemoteDataStore().queryComments(commentQueryRequest).toV2()
            .flatMap {
                // hack before moving to paging data
                if (options.token.isNullOrEmpty()) {
                    return@flatMap clearComments()
                        .andThen(Single.defer { Single.just(it) })
                } else {
                    return@flatMap Single.just(it)
                }
            }
            .flatMap {
                Single.defer { CommentQueryPersister().persist(it).toV2().andThen(Single.just(it)) }
            }
            .doOnSuccess { dto: EkoCommentAndUserListDto ->
                val comments = dto.comments
                if (!comments.isNullOrEmpty()) {
                    val commentId =
                        comments.lastOrNull { if (this.isDeleted == null) true else this.isDeleted == it.deleted }?.commentId
                    val nextPage = dto.token.next
                    if (!nextPage.isNullOrEmpty()) {
                        if (!commentId.isNullOrEmpty()) {
                            commentIdAndTokenMap[commentId] = Pair.create(nextPage, false)
                            if (commentIdSet.contains(commentId)) {
                                mapByComment(commentId)
                            }
                        } else {
                            val options =
                                CommentQueryRequestOptions(token = nextPage, type = "pagination")
                            call(options)
                                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                                .subscribe(this)
                        }
                    }
                }
            }
            .retryWhen(RetryWhen.maxRetries(3)
                .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                .action { errorAndDuration: ErrorAndDuration ->
                    AmityLog.tag(EkoCommentBoundaryCallback::class.java.name).e(
                        errorAndDuration.throwable(),
                        "an error occurred, back-off for durationMs:%s",
                        errorAndDuration.durationMs()
                    )
                }
                .build())
            .ignoreElement()
    }
}