package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunity
import com.amity.socialcloud.sdk.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.social.community.AmityCommunitySortOption
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.CommunityRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityQueryRequest.CommunityQueryOptions
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class EkoCommunityBoundaryCallback(private val keyword: String,
                                            private val categoryId: String,
                                            private val filter: AmityCommunityFilter,
                                            private val sortBy: AmityCommunitySortOption,
                                            private val isDeleted: Boolean?,
                                            pageSize: Int,
                                            private val delaySubject: Subject<Boolean>)
    : EkoBoundaryCallback<AmityCommunity>(pageSize), Function<AmityCommunity, AmityCommunity> {

    private val communityIdAndTokenMap: MutableMap<String, Pair<String, Boolean>> = mutableMapOf()
    private val communityIdSet = hashSetOf<String>()
    private val TAG = javaClass.name

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }

    init {
        onFirstLoaded()
    }

    override fun onItemAtFrontLoaded(itemAtFront: AmityCommunity) {
        // TODO: 2020-07-7
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityCommunity) {
        // TODO: 2020-07-7
    }

    override fun onZeroItemsLoaded() {
        // TODO: 2020-07-7
    }

    override fun onFirstLoaded() {
        AmityLog.tag(TAG).i("onFirstLoaded")
        val options = CommunityQueryOptions(limit = pageSize)
        call(options)
                .doOnComplete { delaySubject.onComplete() }
                .doOnError { delaySubject.onComplete() }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
    }

    override fun apply(input: AmityCommunity): AmityCommunity {
        communityIdSet.add(input.getCommunityId())
        mapByCommunity(input.getCommunityId())
        return input
    }

    private fun mapByCommunity(communityId: String) {
        communityIdAndTokenMap[communityId]?.let { communityIdAndTokenMapNonNull ->
            if (communityIdAndTokenMapNonNull.first.isNullOrEmpty() || communityIdAndTokenMapNonNull.second == true) {
                return
            }
            val options = CommunityQueryOptions(token = communityIdAndTokenMapNonNull.first)
            call(options)
                    .doOnSubscribe { communityIdAndTokenMap[communityId] = Pair(communityIdAndTokenMapNonNull.first, true) }
                    .doOnError { communityIdAndTokenMap[communityId] = Pair(communityIdAndTokenMapNonNull.first, false) }
                    .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                    .subscribe(this)
        }
    }

    private fun call(options: CommunityQueryOptions): Completable {
        return CommunityRemoteDataStore().queryCommunities(
            keyword = keyword.ifEmpty { null },
            categoryId = categoryId.ifEmpty { null },
            filter = filter.value,
            sortBy = sortBy.apiKey,
            options = QueryOptionsRequestParams(
                limit = options.limit,
                token = options.token
            )
        ).toV2()
            .flatMap { dto ->
                CommunityQueryPersister().persist(dto).toV2()
                    .andThen(Single.just(dto))
            }
                .doOnSuccess { dto: CommunityListQueryDto ->
                    val communities = dto.communities
                    if (!communities.isNullOrEmpty()) {
                        val communityId = communities.lastOrNull { if (this.isDeleted == null) true else this.isDeleted == it.isDeleted }?.communityId
                        val nextPage = dto.paging?.next
                        if (!nextPage.isNullOrEmpty()) {
                            if (!communityId.isNullOrEmpty()) {
                                communityIdAndTokenMap[communityId] = Pair.create(nextPage, false)
                                if (communityIdSet.contains(communityId)) {
                                    mapByCommunity(communityId)
                                }
                            } else {
                                val options = CommunityQueryOptions(token = nextPage)
                                call(options)
                                        .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                                        .subscribe(this)
                            }
                        }
                    }
                }
                .retryWhen(RetryWhen.maxRetries(3)
                        .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                        .action { errorAndDuration: ErrorAndDuration ->
                            AmityLog.tag(TAG).e(errorAndDuration.throwable(),
                                    "an error occurred, back-off for durationMs:%s",
                                    errorAndDuration.durationMs())
                        }
                        .build())
                .ignoreElement()
    }
}