package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunityCategory
import com.amity.socialcloud.sdk.social.community.AmityCommunityCategorySortOption
import com.amity.socialcloud.sdk.social.data.category.CategoryQueryPersister
import com.amity.socialcloud.sdk.social.data.category.CategoryRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.EkoCommunityCategoryListDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityCategoriesQueryRequest.CommunityCategoriesQueryOptions
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class EkoCommunityCategoryBoundaryCallback(
    private val sortOption: AmityCommunityCategorySortOption,
    private val isDeleted: Boolean?,
    pageSize: Int,
    private val delaySubject: Subject<Boolean>
) : EkoBoundaryCallback<AmityCommunityCategory>(pageSize),
    Function<AmityCommunityCategory, AmityCommunityCategory> {

    private val TAG = javaClass.name
    
    private val categoryIdAndTokenMap: MutableMap<String, Pair<String, Boolean>> = mutableMapOf()
    private val categoryIdSet = hashSetOf<String>()

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }


    init {
        onFirstLoaded()
    }

    override fun onItemAtFrontLoaded(itemAtFront: AmityCommunityCategory) {
        // TODO: 2019-08-31
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityCommunityCategory) {
        // TODO: 2019-08-31
    }

    override fun onZeroItemsLoaded() {
        // TODO: 2019-08-31
    }

    override fun onFirstLoaded() {
        AmityLog.tag(TAG).i("onFirstLoaded")
        val options = CommunityCategoriesQueryOptions()
        options.limit = pageSize
        call(options)
            .doOnComplete { delaySubject.onComplete() }
            .doOnError { delaySubject.onComplete() }
            .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
            .subscribe(this)
    }

    override fun apply(input: AmityCommunityCategory): AmityCommunityCategory {
        categoryIdSet.add(input.getCategoryId())
        mapByCategory(input.getCategoryId())
        return input
    }

    private fun mapByCategory(categoryId: String) {
        categoryIdAndTokenMap[categoryId]?.let { categoryIdAndTokenMapNonNull ->
            if (categoryIdAndTokenMapNonNull.first.isNullOrEmpty() || categoryIdAndTokenMapNonNull.second == true) {
                return
            }
            val options = CommunityCategoriesQueryOptions()
            options.token = categoryIdAndTokenMapNonNull.first
            call(options)
                .doOnSubscribe {
                    categoryIdAndTokenMap[categoryId] =
                        Pair(categoryIdAndTokenMapNonNull.first, true)
                }
                .doOnError {
                    categoryIdAndTokenMap[categoryId] =
                        Pair(categoryIdAndTokenMapNonNull.first, false)
                }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
        }
    }

    private fun call(options: CommunityCategoriesQueryOptions): Completable {
        return CategoryRemoteDataStore().queryCategories(
            sortBy = sortOption.apiKey,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(limit = options.limit, token = options.token)
        ).toV2()
            .flatMap { dto ->
                CategoryQueryPersister().persist(dto).toV2()
                    .andThen(Single.just(dto))
            }
            .doOnSuccess { dto: EkoCommunityCategoryListDto ->
                val communityCategories = dto.results
                if (!communityCategories.isNullOrEmpty()) {
                    val categoryId =
                        communityCategories.lastOrNull { if (this.isDeleted == null) true else this.isDeleted == it.deleted }?.categoryId
                    val nextPage = dto.token.next
                    if (!nextPage.isNullOrEmpty()) {
                        if (!categoryId.isNullOrEmpty()) {
                            categoryIdAndTokenMap[categoryId] = Pair.create(nextPage, false)
                            if (categoryIdSet.contains(categoryId)) {
                                mapByCategory(categoryId)
                            }
                        } else {
                            val options = CommunityCategoriesQueryOptions()
                            options.token = nextPage
                            call(options)
                                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                                .subscribe(this)
                        }
                    }
                }
            }
            .retryWhen(RetryWhen.maxRetries(3)
                .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                .action { errorAndDuration: ErrorAndDuration ->
                    AmityLog.tag(TAG).e(
                        errorAndDuration.throwable(),
                        "an error occurred, back-off for durationMs:%s",
                        errorAndDuration.durationMs()
                    )
                }
                .build())
            .ignoreElement()
    }
}
