package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunityMember
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityMembershipQueryRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.CommunityMemberQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.CommunityMemberQueryTokenEntity
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers


// unix page number for backward compat!
private const val PAGE_NUMBER = 1

internal class EkoCommunityMembershipBoundaryCallback(
    private val communityId: String,
    private val memberships: List<String>?,
    private val roles: List<String>?,
    private val sortOption: String,
    val pageSize: Int
) : PagedList.BoundaryCallback<AmityCommunityMember>() {

    init {
        onFirstLoaded()
    }

    private val tokenDao: CommunityMemberQueryTokenDao =
        UserDatabase.get().communityMemberQueryTokenDao()

    private fun onFirstLoaded() {
        val request = CommunityMembershipQueryRequest(
            communityId = communityId,
            memberships = memberships,
            roles = roles,
            sortBy = sortOption,
            options = CommunityMembershipQueryRequest.CommunityMembershipQueryOptions(limit = pageSize)
        )
        call(request)
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityCommunityMember) {
        tokenDao.getQueryTokenByPageNumber(
            primaryKeys = mapOf(
                "communityId" to communityId,
                "memberships" to (memberships?.toString() ?: ""),
                "roles" to (roles?.toString() ?: ""),
                "sortOption" to sortOption
            ), pageNumber = PAGE_NUMBER
        )
            .filter { token: CommunityMemberQueryTokenEntity -> !token.next.isNullOrEmpty() }
            .flatMapCompletable { token: CommunityMemberQueryTokenEntity ->
                call(
                    CommunityMembershipQueryRequest(
                        communityId = communityId,
                        memberships = memberships,
                        roles = roles,
                        sortBy = sortOption,
                        options = CommunityMembershipQueryRequest.CommunityMembershipQueryOptions(
                            token = token.next!!
                        )
                    )
                )
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    private fun call(request: CommunityMembershipQueryRequest): Completable {
        return CommunityMembershipRemoteDataStore().queryCommunityUsers(
            communityId = communityId,
            memberships = memberships,
            roles = roles,
            sortBy = sortOption,
            options = QueryOptionsRequestParams(
                limit = request.options?.limit,
                token = request.options?.token
            )
        ).toV2()
            .flatMap {
                CommunityQueryPersister().persist(it).toV2()
                    .andThen(Single.just(it))
            }
            .doOnSuccess { dto ->
                val token = CommunityMemberQueryTokenEntity().apply {
                    this.communityId = this@EkoCommunityMembershipBoundaryCallback.communityId
                    this.memberships =
                        this@EkoCommunityMembershipBoundaryCallback.memberships?.toString() ?: ""
                    this.roles = this@EkoCommunityMembershipBoundaryCallback.roles?.toString() ?: ""
                    this.sortOption = this@EkoCommunityMembershipBoundaryCallback.sortOption
                    this.pageNumber = PAGE_NUMBER
                    this.next = dto.paging?.next
                    this.previous = dto.paging?.previous
                    this.ids = dto.communityUsers?.map { it.getId() } ?: listOf()
                    this.updatedAt = AmityCoreClient.getServerTime()
                }
                val isFirstPage = dto.paging?.previous == null
                if (isFirstPage) {
                    tokenDao.insertToken(token)
                        .subscribeOn(Schedulers.io())
                        .subscribe()
                } else {
                    updateQueryToken(token)
                }
            }
            .ignoreElement()
    }

    private fun updateQueryToken(token: CommunityMemberQueryTokenEntity) {
        tokenDao.getQueryTokenByPageNumber(
            primaryKeys = mapOf(
                "communityId" to communityId,
                "memberships" to (memberships?.toString() ?: ""),
                "roles" to (roles?.toString() ?: ""),
                "sortOption" to sortOption
            ), pageNumber = PAGE_NUMBER
        ).flatMapCompletable {
            token.ids.addAll(0, it.ids)
            tokenDao.insertToken(token)
        }.subscribeOn(Schedulers.io())
            .subscribe()
    }

}